/*
 * Decompiled with CFR 0.152.
 */
package jalview.io;

import jalview.bin.Console;
import jalview.datamodel.DBRefEntry;
import jalview.datamodel.DBRefSource;
import jalview.datamodel.Mapping;
import jalview.datamodel.Sequence;
import jalview.datamodel.SequenceFeature;
import jalview.datamodel.SequenceI;
import jalview.io.AlignFile;
import jalview.io.CdsData;
import jalview.io.FileParse;
import jalview.util.DBRefUtils;
import jalview.util.DnaUtils;
import jalview.util.MapList;
import jalview.util.MappingUtils;
import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.TreeMap;

public abstract class EMBLLikeFlatFile
extends AlignFile {
    protected static final String LOCATION = "location";
    protected static final String QUOTE = "\"";
    protected static final String DOUBLED_QUOTE = "\"\"";
    protected static final String WHITESPACE = "\\s+";
    protected boolean produceRna = true;
    protected String sourceDb;
    protected String accession;
    protected String version;
    protected String description;
    protected int length = 128;
    protected List<DBRefEntry> dbrefs;
    protected boolean sequenceStringIsRNA = false;
    protected String sequenceString;
    protected Map<String, CdsData> cds;

    protected static String removeQuotes(String value) {
        if (value == null) {
            return null;
        }
        if (value.startsWith(QUOTE) && !value.startsWith(DOUBLED_QUOTE)) {
            value = value.substring(1);
        }
        if (value.endsWith(QUOTE) && !value.endsWith(DOUBLED_QUOTE)) {
            value = value.substring(0, value.length() - 1);
        }
        value = value.replace(DOUBLED_QUOTE, QUOTE);
        return value;
    }

    protected static int[] adjustForProteinLength(int proteinLength, int[] exon) {
        if (proteinLength <= 0 || exon == null) {
            return exon;
        }
        int expectedCdsLength = proteinLength * 3;
        int exonLength = MappingUtils.getLength(Arrays.asList(new int[][]{exon}));
        if (expectedCdsLength >= exonLength) {
            return exon;
        }
        int sxpos = -1;
        int endxon = 0;
        int[] origxon = new int[exon.length];
        System.arraycopy(exon, 0, origxon, 0, exon.length);
        int cdspos = 0;
        for (int x = 0; x < exon.length; x += 2) {
            if (expectedCdsLength > (cdspos += Math.abs(exon[x + 1] - exon[x]) + 1)) continue;
            sxpos = x;
            if (expectedCdsLength != cdspos) {
                // empty if block
            }
            if (exon[x + 1] >= exon[x]) {
                endxon = exon[x + 1] - cdspos + expectedCdsLength;
                break;
            }
            endxon = exon[x + 1] + cdspos - expectedCdsLength;
            break;
        }
        if (sxpos != -1) {
            int[] nxon = new int[sxpos + 2];
            System.arraycopy(exon, 0, nxon, 0, sxpos + 2);
            nxon[sxpos + 1] = endxon;
            exon = nxon;
        }
        return exon;
    }

    public EMBLLikeFlatFile(FileParse fp, String sourceId) throws IOException {
        super(false, fp);
        this.sourceDb = sourceId;
        this.dbrefs = new ArrayList<DBRefEntry>();
        this.cds = new TreeMap<String, CdsData>(String.CASE_INSENSITIVE_ORDER);
        this.parse();
    }

    private String parseSourceQualifiers(String[] tokens) throws IOException {
        if (!"source".equals(tokens[0])) {
            throw new RuntimeException("Not given a 'source' qualifier line");
        }
        StringBuilder sb = new StringBuilder().append(tokens[1]);
        String line = this.parseFeatureQualifier(sb, false);
        while (line != null) {
            int qe;
            if (!line.startsWith("FT    ")) {
                return line;
            }
            int p = line.indexOf("\\mol_type");
            int qs = line.indexOf(QUOTE, p);
            String qualifier = line.substring(qs, qe = line.indexOf(QUOTE, qs + 1)).toLowerCase(Locale.ROOT);
            if (qualifier.indexOf("rna") > -1) {
                this.sequenceStringIsRNA = true;
            }
            if (qualifier.indexOf("dna") > -1) {
                this.sequenceStringIsRNA = false;
            }
            line = this.parseFeatureQualifier(sb, false);
        }
        return line;
    }

    protected String parseCDSFeature(String location) throws IOException {
        CdsData data = new CdsData();
        StringBuilder sb = new StringBuilder().append(location);
        String line = this.parseFeatureQualifier(sb, false);
        data.cdsLocation = sb.toString();
        while (line != null && this.isFeatureContinuationLine(line)) {
            int slashPos = line.indexOf(47);
            if (slashPos == -1) {
                Console.error("Unexpected EMBL line ignored: " + line);
                line = this.nextLine();
                continue;
            }
            int eqPos = line.indexOf(61, slashPos + 1);
            if (eqPos == -1) {
                line = this.nextLine();
                continue;
            }
            String qualifier = line.substring(slashPos + 1, eqPos);
            String value = line.substring(eqPos + 1);
            value = EMBLLikeFlatFile.removeQuotes(value);
            sb = new StringBuilder().append(value);
            boolean asText = !"translation".equals(qualifier);
            line = this.parseFeatureQualifier(sb, asText);
            String featureValue = sb.toString();
            if ("protein_id".equals(qualifier)) {
                data.proteinId = featureValue;
                continue;
            }
            if ("codon_start".equals(qualifier)) {
                try {
                    data.codonStart = Integer.parseInt(featureValue.trim());
                }
                catch (NumberFormatException e) {
                    Console.error("Invalid codon_start in XML for " + this.accession + ": " + e.getMessage());
                }
                continue;
            }
            if ("db_xref".equals(qualifier)) {
                String[] parts = featureValue.split(":");
                if (parts.length != 2) continue;
                String db = parts[0].trim();
                db = DBRefUtils.getCanonicalName(db);
                DBRefEntry dbref = new DBRefEntry(db, "0", parts[1].trim());
                data.xrefs.add(dbref);
                continue;
            }
            if ("product".equals(qualifier)) {
                data.proteinName = featureValue;
                continue;
            }
            if ("translation".equals(qualifier)) {
                data.translation = featureValue;
                continue;
            }
            if ("".equals(featureValue)) continue;
            data.cdsProps.put(qualifier, featureValue);
        }
        if (data.proteinId != null) {
            this.cds.put(data.proteinId, data);
        } else {
            Console.error("Ignoring CDS feature with no protein_id for " + this.sourceDb + ":" + this.accession);
        }
        return line;
    }

    protected abstract boolean isFeatureContinuationLine(String var1);

    @Override
    public String print(SequenceI[] seqs, boolean jvsuffix) {
        return null;
    }

    protected void buildSequence() {
        if (this.accession == null || this.sequenceString == null) {
            Console.error("Failed to parse data from EMBL");
            return;
        }
        String name = this.accession;
        if (this.sourceDb != null) {
            name = this.sourceDb + "|" + name;
        }
        if (this.produceRna && this.sequenceStringIsRNA) {
            this.sequenceString = this.sequenceString.replace('T', 'U').replace('t', 'u');
        }
        Sequence seq = new Sequence(name, this.sequenceString);
        seq.setDescription(this.description);
        DBRefEntry selfRef = new DBRefEntry(this.sourceDb, this.version, this.accession);
        int[] startEnd = new int[]{1, seq.getLength()};
        selfRef.setMap(new Mapping(null, startEnd, startEnd, 1, 1));
        seq.addDBRef(selfRef);
        for (DBRefEntry dbref : this.dbrefs) {
            seq.addDBRef(dbref);
        }
        this.processCDSFeatures(seq);
        seq.deriveSequence();
        this.addSequence(seq);
    }

    protected void processCDSFeatures(SequenceI seq) {
        HashMap<String, SequenceI> proteins = new HashMap<String, SequenceI>();
        for (CdsData data : this.cds.values()) {
            this.processCDSFeature(seq, data, proteins);
        }
    }

    void processCDSFeature(SequenceI dna, CdsData data, Map<String, SequenceI> proteins) {
        int[] exons = this.getCdsRanges(this.accession, data.cdsLocation);
        MapList maplist = this.buildMappingToProtein(dna, exons, data);
        int exonNumber = 0;
        for (int xint = 0; exons != null && xint < exons.length - 1; xint += 2) {
            int exonStart = exons[xint];
            int exonEnd = exons[xint + 1];
            int begin = Math.min(exonStart, exonEnd);
            int end = Math.max(exonStart, exonEnd);
            String desc = String.format("Exon %d for protein EMBLCDS:%s", ++exonNumber, data.proteinId);
            SequenceFeature sf = new SequenceFeature("CDS", desc, begin, end, this.sourceDb);
            for (Map.Entry<String, String> val : data.cdsProps.entrySet()) {
                sf.setValue(val.getKey(), val.getValue());
            }
            sf.setEnaLocation(data.cdsLocation);
            boolean forwardStrand = exonStart <= exonEnd;
            sf.setStrand(forwardStrand ? "+" : "-");
            sf.setPhase(String.valueOf(data.codonStart - 1));
            sf.setValue("exon number", exonNumber);
            sf.setValue("product", data.proteinName);
            dna.addSequenceFeature(sf);
        }
        boolean hasUniprotDbref = false;
        for (DBRefEntry xref : data.xrefs) {
            dna.addDBRef(xref);
            if (!xref.getSource().equals("UNIPROT")) continue;
            SequenceI protein = this.buildProteinProduct(dna, xref, data, proteins);
            Mapping map = new Mapping(protein, maplist);
            map.setMappedFromId(data.proteinId);
            xref.setMap(map);
            DBRefEntry db1 = new DBRefEntry(this.sourceDb, this.version, this.accession);
            db1.setMap(new Mapping(dna, maplist.getInverse()));
            protein.addDBRef(db1);
            hasUniprotDbref = true;
        }
        if (!hasUniprotDbref) {
            SequenceI protein = proteins.get(data.proteinId);
            if (protein == null) {
                protein = new Sequence(data.proteinId, data.translation);
                protein.setDescription(data.proteinName);
                proteins.put(data.proteinId, protein);
            }
            DBRefEntry db1 = new DBRefEntry(DBRefSource.EMBLCDSProduct, this.version, data.proteinId);
            protein.addDBRef(db1);
            DBRefEntry dnaToEmblProteinRef = new DBRefEntry(DBRefSource.EMBLCDSProduct, this.version, data.proteinId);
            Mapping map = new Mapping(protein, maplist);
            map.setMappedFromId(data.proteinId);
            dnaToEmblProteinRef.setMap(map);
            dna.addDBRef(dnaToEmblProteinRef);
        }
    }

    MapList buildMappingToProtein(SequenceI dna, int[] exons, CdsData data) {
        MapList dnaToProteinMapping = null;
        int peptideLength = data.translation.length();
        int[] proteinRange = new int[]{1, peptideLength};
        if (exons != null && exons.length > 0) {
            int[] cdsRanges = EMBLLikeFlatFile.adjustForProteinLength(peptideLength, exons);
            dnaToProteinMapping = new MapList(cdsRanges, proteinRange, 3, 1);
        } else {
            Console.error(String.format("Implementation Notice: EMBLCDS location '%s'not properly supported yet - Making up the CDNA region of (%s:%s)... may be incorrect", data.cdsLocation, this.sourceDb, this.accession));
            int completeCodonsLength = 1 - data.codonStart + dna.getLength();
            int mappedDnaEnd = dna.getEnd();
            if (peptideLength * 3 == completeCodonsLength) {
                Console.warn("Assuming no stop codon at end of cDNA fragment");
                mappedDnaEnd = dna.getEnd();
            } else if ((peptideLength + 1) * 3 == completeCodonsLength) {
                Console.warn("Assuming stop codon at end of cDNA fragment");
                mappedDnaEnd = dna.getEnd() - 3;
            }
            if (mappedDnaEnd != -1) {
                int[] cdsRanges = new int[]{dna.getStart() + (data.codonStart - 1), mappedDnaEnd};
                dnaToProteinMapping = new MapList(cdsRanges, proteinRange, 3, 1);
            }
        }
        return dnaToProteinMapping;
    }

    SequenceI buildProteinProduct(SequenceI dna, DBRefEntry xref, CdsData data, Map<String, SequenceI> proteins) {
        if (data.proteinId == null || data.translation == null) {
            return null;
        }
        String proteinSeqName = xref.getSource() + "|" + xref.getAccessionId();
        SequenceI protein = proteins.get(proteinSeqName);
        if (protein == null) {
            protein = new Sequence(proteinSeqName, data.translation, 1, data.translation.length());
            protein.setDescription(data.proteinName != null ? data.proteinName : "Protein Product from " + this.sourceDb);
            proteins.put(proteinSeqName, protein);
        }
        return protein;
    }

    protected int[] getCdsRanges(String accession, String location) {
        if (location == null) {
            return new int[0];
        }
        try {
            List<int[]> ranges = DnaUtils.parseLocation(location);
            return MappingUtils.rangeListToArray(ranges);
        }
        catch (ParseException e) {
            Console.warn(String.format("Not parsing inexact CDS location %s in ENA %s", location, accession));
            return new int[0];
        }
    }

    String parseFeatureQualifier(StringBuilder sb, boolean asText) throws IOException {
        String line;
        while ((line = this.nextLine()) != null && this.isFeatureContinuationLine(line)) {
            String[] tokens = line.split(WHITESPACE);
            if (tokens.length < 2) {
                Console.error("Ignoring bad EMBL line for " + this.accession + ": " + line);
                break;
            }
            if (tokens[1].startsWith("/")) break;
            if (asText) {
                sb.append(" ");
            }
            String data = EMBLLikeFlatFile.removeQuotes(tokens[1]);
            sb.append(data);
        }
        return line;
    }

    protected String parseSequence() throws IOException {
        StringBuilder sb = new StringBuilder(this.length);
        String line = this.nextLine();
        while (line != null && line.startsWith(" ")) {
            line = line.trim();
            String[] blocks = line.split(WHITESPACE);
            for (int i = 0; i < blocks.length; ++i) {
                try {
                    Long.parseLong(blocks[i]);
                    continue;
                }
                catch (NumberFormatException e) {
                    sb.append(blocks[i]);
                }
            }
            line = this.nextLine();
        }
        this.sequenceString = sb.toString();
        return line;
    }

    protected String parseFeature(String line) throws IOException {
        String[] tokens = line.trim().split(WHITESPACE);
        if (tokens.length < 2 || !"CDS".equals(tokens[0]) && !"source".equals(tokens[0])) {
            return this.nextLine();
        }
        if (tokens[0].equals("source")) {
            return this.parseSourceQualifiers(tokens);
        }
        return this.parseCDSFeature(tokens[1]);
    }
}

