/*
 * Decompiled with CFR 0.152.
 */
package jalview.analysis;

import jalview.analysis.AlignmentAnnotationUtils;
import jalview.analysis.CodonComparator;
import jalview.analysis.Dna;
import jalview.analysis.SequenceIdMatcher;
import jalview.bin.Console;
import jalview.commands.RemoveGapColCommand;
import jalview.datamodel.AlignedCodon;
import jalview.datamodel.AlignedCodonFrame;
import jalview.datamodel.Alignment;
import jalview.datamodel.AlignmentAnnotation;
import jalview.datamodel.AlignmentI;
import jalview.datamodel.Annotation;
import jalview.datamodel.ContactMatrixI;
import jalview.datamodel.DBRefEntry;
import jalview.datamodel.GeneLociI;
import jalview.datamodel.IncompleteCodonException;
import jalview.datamodel.Mapping;
import jalview.datamodel.Sequence;
import jalview.datamodel.SequenceFeature;
import jalview.datamodel.SequenceGroup;
import jalview.datamodel.SequenceI;
import jalview.datamodel.features.SequenceFeatures;
import jalview.schemes.ResidueProperties;
import jalview.util.ColorUtils;
import jalview.util.Comparison;
import jalview.util.Constants;
import jalview.util.DBRefUtils;
import jalview.util.IntRangeComparator;
import jalview.util.MapList;
import jalview.util.MappingUtils;
import jalview.util.ShiftList;
import java.awt.Color;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.stream.Collectors;

public class AlignmentUtils {
    private static final int CODON_LENGTH = 3;
    private static final String SEQUENCE_VARIANT = "sequence_variant:";
    public static final String VARIANT_ID = "id";

    public static AlignmentI expandContext(AlignmentI core, int flankSize) {
        ArrayList<SequenceI> sq = new ArrayList<SequenceI>();
        int maxoffset = 0;
        for (SequenceI s : core.getSequences()) {
            SequenceI newSeq = s.deriveSequence();
            int newSeqStart = newSeq.getStart() - 1;
            if (newSeqStart > maxoffset && newSeq.getDatasetSequence().getStart() < s.getStart()) {
                maxoffset = newSeqStart;
            }
            sq.add(newSeq);
        }
        if (flankSize > -1) {
            maxoffset = Math.min(maxoffset, flankSize);
        }
        Iterator<SequenceI> iterator = sq.iterator();
        while (iterator.hasNext()) {
            int p;
            SequenceI s;
            SequenceI ds = s = iterator.next();
            while (ds.getDatasetSequence() != null) {
                ds = ds.getDatasetSequence();
            }
            int s_end = s.findPosition(s.getStart() + s.getLength());
            int ustream_ds = s.getStart() - ds.getStart();
            int dstream_ds = ds.getEnd() - s_end;
            int offset = maxoffset - ustream_ds;
            if (flankSize >= 0) {
                if (flankSize < ustream_ds) {
                    offset = maxoffset - flankSize;
                    ustream_ds = flankSize;
                }
                if (flankSize <= dstream_ds) {
                    dstream_ds = flankSize - 1;
                }
            }
            char[] upstream = new String(ds.getSequence(s.getStart() - 1 - ustream_ds, s.getStart() - 1)).toLowerCase(Locale.ROOT).toCharArray();
            char[] downstream = new String(ds.getSequence(s_end - 1, s_end + dstream_ds)).toLowerCase(Locale.ROOT).toCharArray();
            char[] coreseq = s.getSequence();
            char[] nseq = new char[offset + upstream.length + downstream.length + coreseq.length];
            char c = core.getGapCharacter();
            for (p = 0; p < offset; ++p) {
                nseq[p] = c;
            }
            System.arraycopy(upstream, 0, nseq, p, upstream.length);
            System.arraycopy(coreseq, 0, nseq, p + upstream.length, coreseq.length);
            System.arraycopy(downstream, 0, nseq, p + coreseq.length + upstream.length, downstream.length);
            s.setSequence(new String(nseq));
            s.setStart(s.getStart() - ustream_ds);
            s.setEnd(s_end + downstream.length);
        }
        Alignment newAl = new Alignment(sq.toArray(new SequenceI[0]));
        for (SequenceI s : sq) {
            if (s.getAnnotation() == null) continue;
            for (AlignmentAnnotation aa : s.getAnnotation()) {
                aa.adjustForAlignment();
                newAl.addAnnotation(aa);
            }
        }
        newAl.setDataset(core.getDataset());
        return newAl;
    }

    public static int getSequenceIndex(AlignmentI al, SequenceI seq) {
        int result = -1;
        int pos = 0;
        for (SequenceI alSeq : al.getSequences()) {
            if (alSeq == seq) {
                result = pos;
                break;
            }
            ++pos;
        }
        return result;
    }

    public static Map<String, List<SequenceI>> getSequencesByName(AlignmentI al) {
        LinkedHashMap<String, List<SequenceI>> theMap = new LinkedHashMap<String, List<SequenceI>>();
        for (SequenceI seq : al.getSequences()) {
            String name = seq.getName();
            if (name == null) continue;
            ArrayList<SequenceI> seqs = (ArrayList<SequenceI>)theMap.get(name);
            if (seqs == null) {
                seqs = new ArrayList<SequenceI>();
                theMap.put(name, seqs);
            }
            seqs.add(seq);
        }
        return theMap;
    }

    public static boolean mapProteinAlignmentToCdna(AlignmentI proteinAlignment, AlignmentI cdnaAlignment) {
        if (proteinAlignment == null || cdnaAlignment == null) {
            return false;
        }
        HashSet<SequenceI> mappedDna = new HashSet<SequenceI>();
        HashSet<SequenceI> mappedProtein = new HashSet<SequenceI>();
        boolean mappingPerformed = AlignmentUtils.mapProteinToCdna(proteinAlignment, cdnaAlignment, mappedDna, mappedProtein, true);
        return mappingPerformed |= AlignmentUtils.mapProteinToCdna(proteinAlignment, cdnaAlignment, mappedDna, mappedProtein, false);
    }

    protected static boolean mapProteinToCdna(AlignmentI proteinAlignment, AlignmentI cdnaAlignment, Set<SequenceI> mappedDna, Set<SequenceI> mappedProtein, boolean xrefsOnly) {
        boolean mappingExistsOrAdded = false;
        List<SequenceI> thisSeqs = proteinAlignment.getSequences();
        for (SequenceI aaSeq : thisSeqs) {
            boolean proteinMapped = false;
            AlignedCodonFrame acf = new AlignedCodonFrame();
            for (SequenceI cdnaSeq : cdnaAlignment.getSequences()) {
                if (xrefsOnly && !AlignmentUtils.haveCrossRef(aaSeq, cdnaSeq) || !xrefsOnly && (mappedProtein.contains(aaSeq) || mappedDna.contains(cdnaSeq))) continue;
                if (AlignmentUtils.mappingExists(proteinAlignment.getCodonFrames(), aaSeq.getDatasetSequence(), cdnaSeq.getDatasetSequence())) {
                    mappingExistsOrAdded = true;
                    continue;
                }
                MapList map = AlignmentUtils.mapCdnaToProtein(aaSeq, cdnaSeq);
                if (map == null) continue;
                acf.addMap(cdnaSeq, aaSeq, map);
                mappingExistsOrAdded = true;
                proteinMapped = true;
                mappedDna.add(cdnaSeq);
                mappedProtein.add(aaSeq);
            }
            if (!proteinMapped) continue;
            proteinAlignment.addCodonFrame(acf);
        }
        return mappingExistsOrAdded;
    }

    protected static boolean mappingExists(List<AlignedCodonFrame> mappings, SequenceI aaSeq, SequenceI cdnaSeq) {
        if (mappings != null) {
            for (AlignedCodonFrame acf : mappings) {
                if (cdnaSeq != acf.getDnaForAaSeq(aaSeq)) continue;
                return true;
            }
        }
        return false;
    }

    public static MapList mapCdnaToProtein(SequenceI proteinSeq, SequenceI cdnaSeq) {
        char[] cdnaSeqChars;
        SequenceI proteinDataset = proteinSeq.getDatasetSequence();
        char[] aaSeqChars = proteinDataset != null ? proteinDataset.getSequence() : proteinSeq.getSequence();
        SequenceI cdnaDataset = cdnaSeq.getDatasetSequence();
        char[] cArray = cdnaSeqChars = cdnaDataset != null ? cdnaDataset.getSequence() : cdnaSeq.getSequence();
        if (aaSeqChars == null || cdnaSeqChars == null) {
            return null;
        }
        int mappedLength = 3 * aaSeqChars.length;
        int cdnaLength = cdnaSeqChars.length;
        int cdnaStart = cdnaSeq.getStart();
        int cdnaEnd = cdnaSeq.getEnd();
        int proteinStart = proteinSeq.getStart();
        int proteinEnd = proteinSeq.getEnd();
        if (cdnaLength != mappedLength && cdnaLength > 2) {
            String lastCodon = String.valueOf(cdnaSeqChars, cdnaLength - 3, 3).toUpperCase(Locale.ROOT);
            for (String stop : ResidueProperties.STOP_CODONS) {
                if (!lastCodon.equals(stop)) continue;
                cdnaEnd -= 3;
                cdnaLength -= 3;
                break;
            }
        }
        int startOffset = 0;
        if (cdnaLength != mappedLength && cdnaLength > 2 && String.valueOf(cdnaSeqChars, 0, 3).toUpperCase(Locale.ROOT).equals(ResidueProperties.START)) {
            startOffset += 3;
            cdnaStart += 3;
            cdnaLength -= 3;
        }
        if (AlignmentUtils.translatesAs(cdnaSeqChars, startOffset, aaSeqChars)) {
            MapList map = new MapList(new int[]{cdnaStart, cdnaEnd}, new int[]{proteinStart, proteinEnd}, 3, 1);
            return map;
        }
        return AlignmentUtils.mapCdsToProtein(cdnaSeq, proteinSeq);
    }

    protected static boolean translatesAs(char[] cdnaSeqChars, int cdnaStart, char[] aaSeqChars) {
        String codon;
        int dnaPos;
        if (cdnaSeqChars == null || aaSeqChars == null) {
            return false;
        }
        int aaPos = 0;
        for (dnaPos = cdnaStart; dnaPos < cdnaSeqChars.length - 2 && aaPos < aaSeqChars.length; dnaPos += 3, ++aaPos) {
            codon = String.valueOf(cdnaSeqChars, dnaPos, 3);
            String translated = ResidueProperties.codonTranslate(codon);
            char aaRes = aaSeqChars[aaPos];
            if ((translated == null || ResidueProperties.STOP.equals(translated)) && aaRes == '*' || translated != null && aaRes == translated.charAt(0)) continue;
            return false;
        }
        if (aaPos != aaSeqChars.length) {
            return false;
        }
        if (dnaPos == cdnaSeqChars.length) {
            return true;
        }
        return dnaPos == cdnaSeqChars.length - 3 && ResidueProperties.STOP.equals(ResidueProperties.codonTranslate(codon = String.valueOf(cdnaSeqChars, dnaPos, 3)));
    }

    public static boolean alignSequenceAs(SequenceI seq, AlignmentI al, String gap, boolean preserveMappedGaps, boolean preserveUnmappedGaps) {
        List<AlignedCodonFrame> mappings = al.getCodonFrame(seq);
        if (mappings == null || mappings.isEmpty()) {
            return false;
        }
        SequenceI alignFrom = null;
        AlignedCodonFrame mapping = null;
        for (AlignedCodonFrame mp : mappings) {
            alignFrom = mp.findAlignedSequence(seq, al);
            if (alignFrom == null) continue;
            mapping = mp;
            break;
        }
        if (alignFrom == null) {
            return false;
        }
        AlignmentUtils.alignSequenceAs(seq, alignFrom, mapping, gap, al.getGapCharacter(), preserveMappedGaps, preserveUnmappedGaps);
        return true;
    }

    public static void alignSequenceAs(SequenceI alignTo, SequenceI alignFrom, AlignedCodonFrame mapping, String myGap, char sourceGap, boolean preserveMappedGaps, boolean preserveUnmappedGaps) {
        int thisSeqPos = 0;
        int sourceDsPos = 0;
        int basesWritten = 0;
        char myGapChar = myGap.charAt(0);
        int ratio = myGap.length();
        int fromOffset = alignFrom.getStart() - 1;
        int toOffset = alignTo.getStart() - 1;
        int sourceGapMappedLength = 0;
        boolean inExon = false;
        int toLength = alignTo.getLength();
        int fromLength = alignFrom.getLength();
        StringBuilder thisAligned = new StringBuilder(2 * toLength);
        for (int i = 0; i < fromLength; ++i) {
            int[] mappedPos;
            char sourceChar = alignFrom.getCharAt(i);
            if (sourceChar == sourceGap) {
                sourceGapMappedLength += ratio;
                continue;
            }
            if ((mappedPos = mapping.getMappedRegion(alignTo, alignFrom, ++sourceDsPos + fromOffset)) == null) {
                sourceGapMappedLength += ratio;
                continue;
            }
            int mappedCodonStart = mappedPos[0];
            int mappedCodonEnd = mappedPos[mappedPos.length - 1];
            StringBuilder trailingCopiedGap = new StringBuilder();
            int intronLength = 0;
            while (basesWritten + toOffset < mappedCodonEnd && thisSeqPos < toLength) {
                char c;
                if ((c = alignTo.getCharAt(thisSeqPos++)) != myGapChar) {
                    int sourcePosition;
                    if ((sourcePosition = ++basesWritten + toOffset) < mappedCodonStart) {
                        if (preserveUnmappedGaps && trailingCopiedGap.length() > 0) {
                            thisAligned.append(trailingCopiedGap.toString());
                            intronLength += trailingCopiedGap.length();
                            trailingCopiedGap = new StringBuilder();
                        }
                        ++intronLength;
                        inExon = false;
                    } else {
                        boolean startOfCodon = sourcePosition == mappedCodonStart;
                        int gapsToAdd = AlignmentUtils.calculateGapsToInsert(preserveMappedGaps, preserveUnmappedGaps, sourceGapMappedLength, inExon, trailingCopiedGap.length(), intronLength, startOfCodon);
                        for (int k = 0; k < gapsToAdd; ++k) {
                            thisAligned.append(myGapChar);
                        }
                        sourceGapMappedLength = 0;
                        inExon = true;
                    }
                    thisAligned.append(c);
                    trailingCopiedGap = new StringBuilder();
                    continue;
                }
                if (inExon && preserveMappedGaps) {
                    trailingCopiedGap.append(myGapChar);
                    continue;
                }
                if (inExon || !preserveUnmappedGaps) continue;
                trailingCopiedGap.append(myGapChar);
            }
        }
        while (thisSeqPos < toLength) {
            char c;
            if ((c = alignTo.getCharAt(thisSeqPos++)) != myGapChar || preserveUnmappedGaps) {
                thisAligned.append(c);
            }
            --sourceGapMappedLength;
        }
        if (preserveUnmappedGaps) {
            while (sourceGapMappedLength > 0) {
                thisAligned.append(myGapChar);
                --sourceGapMappedLength;
            }
        }
        alignTo.setSequence(new String(thisAligned));
    }

    protected static int calculateGapsToInsert(boolean preserveMappedGaps, boolean preserveUnmappedGaps, int sourceGapMappedLength, boolean inExon, int trailingGapLength, int intronLength, boolean startOfCodon) {
        int gapsToAdd = 0;
        if (startOfCodon) {
            if (inExon && !preserveMappedGaps) {
                trailingGapLength = 0;
            }
            if (!(inExon || preserveMappedGaps && preserveUnmappedGaps)) {
                trailingGapLength = 0;
            }
            gapsToAdd = inExon ? Math.max(sourceGapMappedLength, trailingGapLength) : (intronLength + trailingGapLength <= sourceGapMappedLength ? sourceGapMappedLength - intronLength : Math.min(intronLength + trailingGapLength - sourceGapMappedLength, trailingGapLength));
        } else {
            if (!preserveMappedGaps) {
                trailingGapLength = 0;
            }
            gapsToAdd = Math.max(sourceGapMappedLength, trailingGapLength);
        }
        return gapsToAdd;
    }

    public static int alignProteinAsDna(AlignmentI protein, AlignmentI dna) {
        if (protein.isNucleotide() || !dna.isNucleotide()) {
            Console.errPrintln("Wrong alignment type in alignProteinAsDna");
            return 0;
        }
        ArrayList<SequenceI> unmappedProtein = new ArrayList<SequenceI>();
        Map<AlignedCodon, Map<SequenceI, AlignedCodon>> alignedCodons = AlignmentUtils.buildCodonColumnsMap(protein, dna, unmappedProtein);
        return AlignmentUtils.alignProteinAs(protein, alignedCodons, unmappedProtein);
    }

    public static int alignCdsAsProtein(AlignmentI dna, AlignmentI protein) {
        if (protein.isNucleotide() || !dna.isNucleotide()) {
            Console.errPrintln("Wrong alignment type in alignProteinAsDna");
            return 0;
        }
        List<AlignedCodonFrame> mappings = protein.getCodonFrames();
        int alignedCount = 0;
        int width = 0;
        for (SequenceI dnaSeq : dna.getSequences()) {
            if (AlignmentUtils.alignCdsSequenceAsProtein(dnaSeq, protein, mappings, dna.getGapCharacter())) {
                ++alignedCount;
            }
            width = Math.max(dnaSeq.getLength(), width);
        }
        for (SequenceI dnaSeq : dna.getSequences()) {
            int oldwidth = dnaSeq.getLength();
            int diff = width - oldwidth;
            if (diff <= 0) continue;
            dnaSeq.insertCharAt(oldwidth, diff, dna.getGapCharacter());
        }
        return alignedCount;
    }

    static boolean alignCdsSequenceAsProtein(SequenceI cdsSeq, AlignmentI protein, List<AlignedCodonFrame> mappings, char gapChar) {
        SequenceI cdsDss = cdsSeq.getDatasetSequence();
        if (cdsDss == null) {
            System.err.println("alignCdsSequenceAsProtein needs aligned sequence!");
            return false;
        }
        List<AlignedCodonFrame> dnaMappings = MappingUtils.findMappingsForSequence(cdsSeq, mappings);
        for (AlignedCodonFrame mapping : dnaMappings) {
            boolean addStopCodon;
            SequenceI peptide = mapping.findAlignedSequence(cdsSeq, protein);
            if (peptide == null) continue;
            int peptideLength = peptide.getLength();
            Mapping map = mapping.getMappingBetween(cdsSeq, peptide);
            if (map == null) continue;
            MapList mapList = map.getMap();
            if (map.getTo() == peptide.getDatasetSequence()) {
                mapList = mapList.getInverse();
            }
            int cdsLength = cdsDss.getLength();
            int mappedFromLength = MappingUtils.getLength(mapList.getFromRanges());
            int mappedToLength = MappingUtils.getLength(mapList.getToRanges());
            boolean bl = addStopCodon = cdsLength == mappedFromLength * 3 + 3 || peptide.getDatasetSequence().getLength() == mappedFromLength - 1;
            if (cdsLength != mappedToLength && !addStopCodon) {
                Console.errPrintln(String.format("Can't align cds as protein (length mismatch %d/%d): %s", cdsLength, mappedToLength, cdsSeq.getName()));
            }
            char[] alignedCds = new char[peptideLength * 3 + (addStopCodon ? 3 : 0)];
            Arrays.fill(alignedCds, gapChar);
            int copiedBases = 0;
            int cdsStart = cdsDss.getStart();
            int proteinPos = peptide.getStart() - 1;
            int cdsCol = 0;
            for (int col = 0; col < peptideLength; ++col) {
                int[] codon;
                char residue = peptide.getCharAt(col);
                if (Comparison.isGap(residue)) {
                    cdsCol += 3;
                    continue;
                }
                if ((codon = mapList.locateInTo(++proteinPos, proteinPos)) == null) {
                    cdsCol += 3;
                    continue;
                }
                for (int j = codon[0]; j <= codon[1]; ++j) {
                    char mappedBase = cdsDss.getCharAt(j - cdsStart);
                    alignedCds[cdsCol++] = mappedBase;
                    ++copiedBases;
                }
            }
            if (copiedBases == cdsLength - 3) {
                int i;
                for (i = alignedCds.length - 1; i >= 0; --i) {
                    if (Comparison.isGap(alignedCds[i])) continue;
                    cdsCol = i + 1;
                    break;
                }
                for (i = cdsLength - 3; i < cdsLength; ++i) {
                    alignedCds[cdsCol++] = cdsDss.getCharAt(i);
                }
            }
            cdsSeq.setSequence(new String(alignedCds));
            return true;
        }
        return false;
    }

    protected static Map<AlignedCodon, Map<SequenceI, AlignedCodon>> buildCodonColumnsMap(AlignmentI protein, AlignmentI dna, List<SequenceI> unmappedProtein) {
        unmappedProtein.addAll(protein.getSequences());
        List<AlignedCodonFrame> mappings = protein.getCodonFrames();
        TreeMap<AlignedCodon, Map<SequenceI, AlignedCodon>> alignedCodons = new TreeMap<AlignedCodon, Map<SequenceI, AlignedCodon>>(new CodonComparator());
        for (SequenceI dnaSeq : dna.getSequences()) {
            for (AlignedCodonFrame mapping : mappings) {
                SequenceI prot = mapping.findAlignedSequence(dnaSeq, protein);
                if (prot == null) continue;
                Mapping seqMap = mapping.getMappingForSequence(dnaSeq);
                AlignmentUtils.addCodonPositions(dnaSeq, prot, protein.getGapCharacter(), seqMap, alignedCodons);
                unmappedProtein.remove(prot);
            }
        }
        int mappedSequenceCount = protein.getHeight() - unmappedProtein.size();
        AlignmentUtils.addUnmappedPeptideStarts(alignedCodons, mappedSequenceCount);
        return alignedCodons;
    }

    protected static void addUnmappedPeptideStarts(Map<AlignedCodon, Map<SequenceI, AlignedCodon>> alignedCodons, int mappedSequenceCount) {
        ArrayList<SequenceI> sequencesChecked = new ArrayList<SequenceI>();
        AlignedCodon lastCodon = null;
        HashMap<SequenceI, AlignedCodon> toAdd = new HashMap<SequenceI, AlignedCodon>();
        for (Map.Entry<AlignedCodon, Map<SequenceI, AlignedCodon>> entry : alignedCodons.entrySet()) {
            for (Map.Entry<SequenceI, AlignedCodon> sequenceCodon : entry.getValue().entrySet()) {
                SequenceI seq = sequenceCodon.getKey();
                if (sequencesChecked.contains(seq)) continue;
                sequencesChecked.add(seq);
                AlignedCodon codon = sequenceCodon.getValue();
                if (codon.peptideCol > 1) {
                    Console.errPrintln("Problem mapping protein with >1 unmapped start positions: " + seq.getName());
                } else if (codon.peptideCol == 1) {
                    AlignedCodon firstPeptide;
                    if (lastCodon != null) {
                        firstPeptide = new AlignedCodon(lastCodon.pos1, lastCodon.pos2, lastCodon.pos3, String.valueOf(seq.getCharAt(0)), 0);
                        toAdd.put(seq, firstPeptide);
                    } else {
                        firstPeptide = new AlignedCodon(0, 0, 0, String.valueOf(seq.getCharAt(0)), 0);
                        toAdd.put(seq, firstPeptide);
                    }
                }
                if (sequencesChecked.size() != mappedSequenceCount) continue;
                break;
            }
            lastCodon = entry.getKey();
        }
        for (Map.Entry<AlignedCodon, Map<SequenceI, AlignedCodon>> entry : toAdd.entrySet()) {
            AlignmentUtils.addCodonToMap(alignedCodons, (AlignedCodon)((Object)entry.getValue()), (SequenceI)((Object)entry.getKey()));
        }
    }

    protected static int alignProteinAs(AlignmentI protein, Map<AlignedCodon, Map<SequenceI, AlignedCodon>> alignedCodons, List<SequenceI> unmappedProtein) {
        int alignedWidth = alignedCodons.size();
        char[] gaps = new char[alignedWidth];
        Arrays.fill(gaps, protein.getGapCharacter());
        HashMap<SequenceI, char[]> peptides = new HashMap<SequenceI, char[]>();
        for (SequenceI seq : protein.getSequences()) {
            if (unmappedProtein.contains(seq)) continue;
            peptides.put(seq, Arrays.copyOf(gaps, gaps.length));
        }
        int column = 0;
        for (AlignedCodon alignedCodon : alignedCodons.keySet()) {
            Map<SequenceI, AlignedCodon> columnResidues = alignedCodons.get(alignedCodon);
            for (Map.Entry<SequenceI, AlignedCodon> entry : columnResidues.entrySet()) {
                char residue;
                ((char[])peptides.get((Object)entry.getKey()))[column] = residue = entry.getValue().product.charAt(0);
            }
            ++column;
        }
        for (Map.Entry entry : peptides.entrySet()) {
            ((SequenceI)entry.getKey()).setSequence(new String((char[])entry.getValue()));
        }
        return 0;
    }

    static void addCodonPositions(SequenceI dna, SequenceI protein, char gapChar, Mapping seqMap, Map<AlignedCodon, Map<SequenceI, AlignedCodon>> alignedCodons) {
        Iterator<AlignedCodon> codons = seqMap.getCodonIterator(dna, gapChar);
        while (codons.hasNext()) {
            try {
                AlignedCodon codon = codons.next();
                AlignmentUtils.addCodonToMap(alignedCodons, codon, protein);
            }
            catch (IncompleteCodonException incompleteCodonException) {
            }
            catch (NoSuchElementException noSuchElementException) {}
        }
    }

    protected static void addCodonToMap(Map<AlignedCodon, Map<SequenceI, AlignedCodon>> alignedCodons, AlignedCodon codon, SequenceI protein) {
        Map<SequenceI, AlignedCodon> seqProduct = alignedCodons.get(codon);
        if (seqProduct == null) {
            seqProduct = new HashMap<SequenceI, AlignedCodon>();
            alignedCodons.put(codon, seqProduct);
        }
        seqProduct.put(protein, codon);
    }

    public static boolean isMappable(AlignmentI al1, AlignmentI al2) {
        if (al1 == null || al2 == null) {
            return false;
        }
        if (al1.isNucleotide() == al2.isNucleotide()) {
            return false;
        }
        AlignmentI dna = al1.isNucleotide() ? al1 : al2;
        AlignmentI protein = dna == al1 ? al2 : al1;
        List<AlignedCodonFrame> mappings = protein.getCodonFrames();
        for (SequenceI dnaSeq : dna.getSequences()) {
            for (SequenceI proteinSeq : protein.getSequences()) {
                if (!AlignmentUtils.isMappable(dnaSeq, proteinSeq, mappings)) continue;
                return true;
            }
        }
        return false;
    }

    protected static boolean isMappable(SequenceI dnaSeq, SequenceI proteinSeq, List<AlignedCodonFrame> mappings) {
        if (dnaSeq == null || proteinSeq == null) {
            return false;
        }
        SequenceI dnaDs = dnaSeq.getDatasetSequence() == null ? dnaSeq : dnaSeq.getDatasetSequence();
        SequenceI proteinDs = proteinSeq.getDatasetSequence() == null ? proteinSeq : proteinSeq.getDatasetSequence();
        for (AlignedCodonFrame mapping : mappings) {
            if (proteinDs != mapping.getAaForDnaSeq(dnaDs)) continue;
            return true;
        }
        return AlignmentUtils.mapCdnaToProtein(proteinDs, dnaDs) != null;
    }

    public static void findAddableReferenceAnnotations(List<SequenceI> sequenceScope, Map<String, String> labelForCalcId, Map<SequenceI, List<AlignmentAnnotation>> candidates, AlignmentI al) {
        if (sequenceScope == null) {
            return;
        }
        for (SequenceI seq : sequenceScope) {
            AlignmentAnnotation[] datasetAnnotations;
            SequenceI dataset = seq.getDatasetSequence();
            if (dataset == null || (datasetAnnotations = dataset.getAnnotation()) == null) continue;
            ArrayList<AlignmentAnnotation> result = new ArrayList<AlignmentAnnotation>();
            for (AlignmentAnnotation dsann : datasetAnnotations) {
                Iterable<AlignmentAnnotation> matchedAlignmentAnnotations = al.findAnnotations(seq, dsann.getCalcId(), dsann.label);
                boolean found = false;
                if (matchedAlignmentAnnotations != null) {
                    for (AlignmentAnnotation matched : matchedAlignmentAnnotations) {
                        if (!dsann.description.equals(matched.description)) continue;
                        found = true;
                        break;
                    }
                }
                if (found) continue;
                result.add(dsann);
                if (labelForCalcId == null) continue;
                labelForCalcId.put(dsann.getCalcId(), dsann.label);
            }
            if (result.isEmpty()) continue;
            candidates.put(seq, result);
        }
    }

    public static void addReferenceAnnotations(Map<SequenceI, List<AlignmentAnnotation>> annotations, AlignmentI alignment, SequenceGroup selectionGroup) {
        for (SequenceI seq : annotations.keySet()) {
            for (AlignmentAnnotation ann : annotations.get(seq)) {
                AlignmentUtils.addReferenceAnnotationTo(alignment, seq, ann, selectionGroup);
            }
        }
    }

    public static boolean isSSAnnotationPresent(Map<SequenceI, List<AlignmentAnnotation>> annotations) {
        for (SequenceI seq : annotations.keySet()) {
            if (!AlignmentUtils.isSecondaryStructurePresent(annotations.get(seq).toArray(new AlignmentAnnotation[0]))) continue;
            return true;
        }
        return false;
    }

    public static AlignmentAnnotation addReferenceAnnotationTo(AlignmentI alignment, SequenceI seq, AlignmentAnnotation ann, SequenceGroup selectionGroup) {
        AlignmentAnnotation copyAnn = new AlignmentAnnotation(ann);
        int startRes = 0;
        int endRes = ann.annotations.length;
        if (selectionGroup != null) {
            startRes = -1 + Math.min(seq.getEnd(), Math.max(seq.getStart(), seq.findPosition(selectionGroup.getStartRes())));
            endRes = -1 + Math.min(seq.getEnd(), seq.findPosition(selectionGroup.getEndRes()));
        }
        copyAnn.restrict(startRes, endRes + 0);
        if (!seq.hasAnnotation(ann)) {
            ContactMatrixI cm = seq.getDatasetSequence().getContactMatrixFor(ann);
            if (cm != null) {
                seq.addContactListFor(copyAnn, cm);
            }
            seq.addAlignmentAnnotation(copyAnn);
        }
        copyAnn.adjustForAlignment();
        alignment.addAnnotation(copyAnn);
        copyAnn.visible = true;
        return copyAnn;
    }

    public static void showOrHideSequenceAnnotations(AlignmentI al, Collection<String> types, List<SequenceI> forSequences, boolean anyType, boolean doShow) {
        AlignmentAnnotation[] anns = al.getAlignmentAnnotation();
        if (anns != null) {
            for (AlignmentAnnotation aa : anns) {
                if (!anyType && !types.contains(aa.label) || aa.sequenceRef == null || forSequences != null && !forSequences.contains(aa.sequenceRef)) continue;
                aa.visible = doShow;
            }
        }
    }

    public static void showOrHideAutoCalculatedAnnotationsForGroup(AlignmentI al, String type, SequenceGroup selectedGroup, boolean anyType, boolean doShow) {
        AlignmentAnnotation[] anns = al.getAlignmentAnnotation();
        if (anns != null) {
            for (AlignmentAnnotation aa : anns) {
                if ((!anyType || !aa.label.startsWith("Secondary Structure Consensus")) && !aa.label.startsWith(type) || aa.groupRef == null || selectedGroup != aa.groupRef) continue;
                aa.visible = doShow;
            }
        }
    }

    public static AlignmentAnnotation getFirstSequenceAnnotationOfType(AlignmentI al, int graphType) {
        AlignmentAnnotation[] anns = al.getAlignmentAnnotation();
        if (anns != null) {
            for (AlignmentAnnotation aa : anns) {
                if (aa.sequenceRef == null || aa.graph != graphType) continue;
                return aa;
            }
        }
        return null;
    }

    public static boolean haveCrossRef(SequenceI seq1, SequenceI seq2) {
        return AlignmentUtils.hasCrossRef(seq1, seq2) || AlignmentUtils.hasCrossRef(seq2, seq1);
    }

    public static boolean hasCrossRef(SequenceI seq1, SequenceI seq2) {
        if (seq1 == null || seq2 == null) {
            return false;
        }
        String name = seq2.getName();
        Sequence.DBModList<DBRefEntry> xrefs = seq1.getDBRefs();
        if (xrefs != null) {
            int nx = xrefs.size();
            for (int ix = 0; ix < nx; ++ix) {
                DBRefEntry xref = (DBRefEntry)xrefs.get(ix);
                String xrefName = xref.getSource() + "|" + xref.getAccessionId();
                if (!xrefName.equalsIgnoreCase(name)) continue;
                return true;
            }
        }
        return false;
    }

    public static AlignmentI makeCdsAlignment(SequenceI[] dna, AlignmentI dataset, SequenceI[] products) {
        if (dataset == null || dataset.getDataset() != null) {
            throw new IllegalArgumentException("IMPLEMENTATION ERROR: dataset.getDataset() must be null!");
        }
        ArrayList<SequenceI> foundSeqs = new ArrayList<SequenceI>();
        ArrayList<SequenceI> cdsSeqs = new ArrayList<SequenceI>();
        List<AlignedCodonFrame> mappings = dataset.getCodonFrames();
        HashSet<SequenceI> productSeqs = null;
        if (products != null) {
            productSeqs = new HashSet<SequenceI>();
            for (SequenceI seq : products) {
                productSeqs.add(seq.getDatasetSequence() == null ? seq : seq.getDatasetSequence());
            }
        }
        for (SequenceI dnaSeq : dna) {
            SequenceI dnaDss = dnaSeq.getDatasetSequence() == null ? dnaSeq : dnaSeq.getDatasetSequence();
            List<AlignedCodonFrame> seqMappings = MappingUtils.findMappingsForSequence(dnaSeq, mappings);
            for (AlignedCodonFrame mapping : seqMappings) {
                List<Mapping> mappingsFromSequence = mapping.getMappingsFromSequence(dnaSeq);
                for (Mapping aMapping : mappingsFromSequence) {
                    MapList mapList = aMapping.getMap();
                    if (mapList.getFromRatio() == 1) continue;
                    SequenceI proteinProduct = aMapping.getTo();
                    if (productSeqs != null && !productSeqs.contains(proteinProduct)) continue;
                    SequenceI cdsSeq = AlignmentUtils.findCdsForProtein(mappings, dnaSeq, seqMappings, aMapping);
                    if (cdsSeq != null) {
                        if (foundSeqs.contains(cdsSeq)) continue;
                        foundSeqs.add(cdsSeq);
                        SequenceI derivedSequence = cdsSeq.deriveSequence();
                        cdsSeqs.add(derivedSequence);
                        if (dataset.getSequences().contains(cdsSeq)) continue;
                        dataset.addSequence(cdsSeq);
                        continue;
                    }
                    cdsSeq = AlignmentUtils.makeCdsSequence(dnaSeq.getDatasetSequence(), aMapping, dataset).deriveSequence();
                    SequenceI cdsSeqDss = cdsSeq.getDatasetSequence();
                    cdsSeqs.add(cdsSeq);
                    List<int[]> cdsRange = Collections.singletonList(new int[]{cdsSeq.getStart(), cdsSeq.getLength() + cdsSeq.getStart() - 1});
                    MapList cdsToProteinMap = new MapList(cdsRange, mapList.getToRanges(), mapList.getFromRatio(), mapList.getToRatio());
                    if (!dataset.getSequences().contains(cdsSeqDss)) {
                        dataset.addSequence(cdsSeqDss);
                        AlignedCodonFrame cdsToProteinMapping = new AlignedCodonFrame();
                        cdsToProteinMapping.addMap(cdsSeqDss, proteinProduct, cdsToProteinMap);
                        if (!mappings.contains(cdsToProteinMapping)) {
                            mappings.add(cdsToProteinMapping);
                        }
                    }
                    AlignmentUtils.propagateDBRefsToCDS(cdsSeqDss, dnaSeq.getDatasetSequence(), proteinProduct, aMapping);
                    AlignedCodonFrame dnaToCdsMapping = new AlignedCodonFrame();
                    MapList dnaToCdsMap = new MapList(mapList.getFromRanges(), cdsRange, 1, 1);
                    dnaToCdsMapping.addMap(dnaSeq.getDatasetSequence(), cdsSeqDss, dnaToCdsMap);
                    if (!mappings.contains(dnaToCdsMapping)) {
                        mappings.add(dnaToCdsMapping);
                    }
                    MapList cdsToDnaMap = dnaToCdsMap.getInverse();
                    AlignmentUtils.transferGeneLoci(dnaSeq, cdsToDnaMap, cdsSeq);
                    List<DBRefEntry> primrefs = dnaDss.getPrimaryDBRefs();
                    int np = primrefs.size();
                    for (int ip = 0; ip < np; ++ip) {
                        DBRefEntry primRef = primrefs.get(ip);
                        String source = primRef.getSource();
                        String version = primRef.getVersion();
                        DBRefEntry cdsCrossRef = new DBRefEntry(source, source + ":" + version, primRef.getAccessionId());
                        cdsCrossRef.setMap(new Mapping(dnaDss, new MapList(cdsToDnaMap)));
                        cdsSeqDss.addDBRef(cdsCrossRef);
                        dnaSeq.addDBRef(new DBRefEntry(source, version, cdsSeq.getName(), new Mapping(cdsSeqDss, dnaToCdsMap)));
                        DBRefEntry proteinToCdsRef = new DBRefEntry(source, version, cdsSeq.getName());
                        proteinToCdsRef.setMap(new Mapping(cdsSeqDss, cdsToProteinMap.getInverse()));
                        proteinProduct.addDBRef(proteinToCdsRef);
                    }
                    AlignmentUtils.transferFeatures(dnaSeq, cdsSeq, dnaToCdsMap, null, "CDS");
                }
            }
        }
        Alignment cds = new Alignment(cdsSeqs.toArray(new SequenceI[cdsSeqs.size()]));
        cds.setDataset(dataset);
        return cds;
    }

    protected static void transferGeneLoci(SequenceI fromSeq, MapList targetToFrom, SequenceI targetSeq) {
        if (targetSeq.getGeneLoci() != null) {
            return;
        }
        GeneLociI fromLoci = fromSeq.getGeneLoci();
        if (fromLoci == null) {
            return;
        }
        MapList newMap = targetToFrom.traverse(fromLoci.getMapping());
        if (newMap != null) {
            targetSeq.setGeneLoci(fromLoci.getSpeciesId(), fromLoci.getAssemblyId(), fromLoci.getChromosomeId(), newMap);
        }
    }

    static SequenceI findCdsForProtein(List<AlignedCodonFrame> mappings, SequenceI dnaSeq, List<AlignedCodonFrame> seqMappings, Mapping aMapping) {
        int dnaLength;
        SequenceI seqDss = dnaSeq.getDatasetSequence() == null ? dnaSeq : dnaSeq.getDatasetSequence();
        SequenceI proteinProduct = aMapping.getTo();
        int mappedFromLength = MappingUtils.getLength(aMapping.getMap().getFromRanges());
        if ((mappedFromLength == (dnaLength = seqDss.getLength()) || mappedFromLength == dnaLength - 3) && seqDss.getFeatures().getFeaturesByOntology("CDS").isEmpty()) {
            return seqDss;
        }
        List<AlignedCodonFrame> mappingsToPeptide = MappingUtils.findMappingsForSequence(proteinProduct, mappings);
        for (AlignedCodonFrame acf : mappingsToPeptide) {
            for (AlignedCodonFrame.SequenceToSequenceMapping map : acf.getMappings()) {
                SequenceI cdsSeq;
                List<AlignedCodonFrame> dnaToCdsMaps;
                Mapping mapping = map.getMapping();
                if (mapping == aMapping || mapping.getMap().getFromRatio() != 3 || proteinProduct != mapping.getTo() || seqDss == map.getFromSeq() || (mappedFromLength = MappingUtils.getLength(mapping.getMap().getFromRanges())) != map.getFromSeq().getLength() || (dnaToCdsMaps = MappingUtils.findMappingsForSequence(cdsSeq = map.getFromSeq(), seqMappings)).isEmpty()) continue;
                return cdsSeq;
            }
        }
        return null;
    }

    static SequenceI makeCdsSequence(SequenceI seq, Mapping mapping, AlignmentI dataset) {
        SequenceI[] matches;
        String mapFromId = mapping.getMappedFromId();
        String seqId = "CDS|" + (mapFromId != null ? mapFromId : seq.getName());
        SequenceI newSeq = null;
        char[] seqChars = seq.getSequence();
        List<int[]> fromRanges = mapping.getMap().getFromRanges();
        int cdsWidth = MappingUtils.getLength(fromRanges);
        char[] newSeqChars = new char[cdsWidth];
        int newPos = 0;
        for (int[] range : fromRanges) {
            if (range[0] <= range[1]) {
                int length = range[1] - range[0] + 1;
                System.arraycopy(seqChars, range[0] - 1, newSeqChars, newPos, length);
                newPos += length;
            } else {
                for (int i = range[0]; i >= range[1]; --i) {
                    newSeqChars[newPos++] = Dna.getComplement(seqChars[i - 1]);
                }
            }
            newSeq = new Sequence(seqId, newSeqChars, 1, newPos);
        }
        if (dataset != null && (matches = dataset.findSequenceMatch(newSeq.getName())) != null) {
            boolean matched = false;
            for (SequenceI mtch : matches) {
                if (mtch.getStart() != newSeq.getStart() || mtch.getEnd() != newSeq.getEnd() || !Arrays.equals(mtch.getSequence(), newSeq.getSequence())) continue;
                if (!matched) {
                    matched = true;
                    newSeq = mtch;
                    continue;
                }
                Console.error("JAL-2154 regression: warning - found (and ignored) a duplicate CDS sequence:" + mtch.toString());
            }
        }
        return newSeq;
    }

    protected static List<DBRefEntry> propagateDBRefsToCDS(SequenceI cdsSeq, SequenceI contig, SequenceI proteinProduct, Mapping mapping) {
        ArrayList<DBRefEntry> direct = new ArrayList<DBRefEntry>();
        HashSet<String> directSources = new HashSet<String>();
        Sequence.DBModList<DBRefEntry> refs = contig.getDBRefs();
        if (refs != null) {
            int nb = refs.size();
            for (int ib = 0; ib < nb; ++ib) {
                MapList map;
                DBRefEntry dbr = (DBRefEntry)refs.get(ib);
                if (!dbr.hasMap() || !(map = dbr.getMap().getMap()).isTripletMap() || !mapping.getMap().equals(map)) continue;
                direct.add(dbr);
                directSources.add(dbr.getSource());
            }
        }
        List<DBRefEntry> onSource = DBRefUtils.selectRefs(proteinProduct.getDBRefs(), directSources.toArray(new String[0]));
        ArrayList<DBRefEntry> propagated = new ArrayList<DBRefEntry>();
        int nc = direct.size();
        for (int ic = 0; ic < nc; ++ic) {
            List<DBRefEntry> sourceRefs;
            DBRefEntry cdsref = (DBRefEntry)direct.get(ic);
            Mapping m = cdsref.getMap();
            MapList cdsposmap = new MapList(Arrays.asList(new int[][]{{cdsSeq.getStart(), cdsSeq.getEnd()}}), m.getMap().getToRanges(), 3, 1);
            Mapping cdsmap = new Mapping(m.getTo(), m.getMap());
            DBRefEntry newref = new DBRefEntry(cdsref.getSource(), cdsref.getVersion(), cdsref.getAccessionId(), new Mapping(cdsmap.getTo(), cdsposmap));
            if (cdsmap.getTo() == null && onSource != null && (sourceRefs = DBRefUtils.searchRefs(onSource, cdsref.getAccessionId())) != null) {
                for (DBRefEntry srcref : sourceRefs) {
                    if (!srcref.getSource().equalsIgnoreCase(cdsref.getSource())) continue;
                    newref.getMap().setTo(proteinProduct);
                }
            }
            cdsSeq.addDBRef(newref);
            propagated.add(newref);
        }
        return propagated;
    }

    protected static int transferFeatures(SequenceI fromSeq, SequenceI toSeq, MapList mapping, String select, String ... omitting) {
        SequenceI copyTo = toSeq;
        while (copyTo.getDatasetSequence() != null) {
            copyTo = copyTo.getDatasetSequence();
        }
        if (fromSeq == copyTo || fromSeq.getDatasetSequence() == copyTo) {
            return 0;
        }
        List<SequenceFeature> sfs = select == null ? fromSeq.getFeatures().getPositionalFeatures(new String[0]) : fromSeq.getFeatures().getFeaturesByOntology(select);
        int count = 0;
        for (SequenceFeature sf : sfs) {
            int end;
            String type = sf.getType();
            boolean omit = false;
            for (String toOmit : omitting) {
                if (!type.equals(toOmit)) continue;
                omit = true;
            }
            if (omit) continue;
            int start = sf.getBegin();
            int[] mappedTo = mapping.locateInTo(start, end = sf.getEnd());
            if (mappedTo == null && (mappedTo = mapping.locateInTo(end, end)) != null) {
                mappedTo[0] = 1;
            }
            if (mappedTo == null && (mappedTo = mapping.locateInTo(start, start)) != null) {
                mappedTo[1] = toSeq.getLength();
            }
            if (mappedTo == null) continue;
            int newBegin = Math.min(mappedTo[0], mappedTo[1]);
            int newEnd = Math.max(mappedTo[0], mappedTo[1]);
            SequenceFeature copy = new SequenceFeature(sf, newBegin, newEnd, sf.getFeatureGroup(), sf.getScore());
            copyTo.addSequenceFeature(copy);
            ++count;
        }
        return count;
    }

    public static MapList mapCdsToProtein(SequenceI dnaSeq, SequenceI proteinSeq) {
        List<int[]> ranges = AlignmentUtils.findCdsPositions(dnaSeq);
        int mappedDnaLength = MappingUtils.getLength(ranges);
        int codonRemainder = mappedDnaLength % 3;
        if (codonRemainder > 0) {
            mappedDnaLength -= codonRemainder;
            MappingUtils.removeEndPositions(codonRemainder, ranges);
        }
        int proteinLength = proteinSeq.getLength();
        int proteinStart = proteinSeq.getStart();
        int proteinEnd = proteinSeq.getEnd();
        if (proteinSeq.getCharAt(0) == 'X') {
            ++proteinStart;
            --proteinLength;
        }
        ArrayList<int[]> proteinRange = new ArrayList<int[]>();
        int codesForResidues = mappedDnaLength / 3;
        if (codesForResidues == proteinLength + 1) {
            --codesForResidues;
            mappedDnaLength -= 3;
            MappingUtils.removeEndPositions(3, ranges);
        }
        if (codesForResidues == proteinLength) {
            proteinRange.add(new int[]{proteinStart, proteinEnd});
            return new MapList(ranges, proteinRange, 3, 1);
        }
        return null;
    }

    protected static List<int[]> findCdsPositions(SequenceI dnaSeq) {
        ArrayList<int[]> result = new ArrayList<int[]>();
        List<SequenceFeature> sfs = dnaSeq.getFeatures().getFeaturesByOntology("CDS");
        if (sfs.isEmpty()) {
            return result;
        }
        SequenceFeatures.sortFeatures(sfs, true);
        for (SequenceFeature sf : sfs) {
            int phase = 0;
            try {
                String s = sf.getPhase();
                if (s != null) {
                    phase = Integer.parseInt(s);
                }
            }
            catch (NumberFormatException s) {
                // empty catch block
            }
            int begin = sf.getBegin();
            int end = sf.getEnd();
            if (result.isEmpty() && phase > 0 && (begin += phase) > end) {
                System.err.println("Error: start phase extends beyond start CDS in " + dnaSeq.getName());
            }
            result.add(new int[]{begin, end});
        }
        Collections.sort(result, IntRangeComparator.ASCENDING);
        return result;
    }

    public static AlignmentI makeCopyAlignment(SequenceI[] seqs, SequenceI[] xrefs, AlignmentI dataset) {
        Alignment copy = new Alignment(new Alignment(seqs));
        copy.setDataset(dataset);
        boolean isProtein = !copy.isNucleotide();
        SequenceIdMatcher matcher = new SequenceIdMatcher(seqs);
        if (xrefs != null) {
            for (SequenceI xref : xrefs) {
                Sequence.DBModList<DBRefEntry> dbrefs = xref.getDBRefs();
                if (dbrefs == null) continue;
                int nir = dbrefs.size();
                for (int ir = 0; ir < nir; ++ir) {
                    SequenceI mappedTo;
                    SequenceI match;
                    SequenceI mto;
                    DBRefEntry dbref = (DBRefEntry)dbrefs.get(ir);
                    Mapping map = dbref.getMap();
                    if (map == null || (mto = map.getTo()) == null || mto.isProtein() != isProtein || (match = matcher.findIdMatch(mappedTo = mto)) != null) continue;
                    matcher.add(mappedTo);
                    copy.addSequence(mappedTo);
                }
            }
        }
        return copy;
    }

    public static int alignAs(AlignmentI unaligned, AlignmentI aligned) {
        if (AlignmentUtils.alignAsSameSequences(unaligned, aligned)) {
            return unaligned.getHeight();
        }
        ArrayList<SequenceI> unmapped = new ArrayList<SequenceI>();
        SortedMap<Integer, Map<SequenceI, Character>> columnMap = AlignmentUtils.buildMappedColumnsMap(unaligned, aligned, unmapped);
        int width = columnMap.size();
        char gap = unaligned.getGapCharacter();
        int realignedCount = 0;
        for (SequenceI seq : unaligned.getSequences()) {
            if (unmapped.contains(seq)) continue;
            char[] newSeq = new char[width];
            Arrays.fill(newSeq, gap);
            int newCol = 0;
            int lastCol = 0;
            for (Integer column : columnMap.keySet()) {
                Character c = (Character)((Map)columnMap.get(column)).get(seq);
                if (c != null) {
                    newSeq[newCol] = c.charValue();
                    lastCol = newCol;
                }
                ++newCol;
            }
            if (lastCol < width) {
                char[] tmp = new char[lastCol + 1];
                System.arraycopy(newSeq, 0, tmp, 0, lastCol + 1);
                newSeq = tmp;
            }
            seq.setSequence(String.valueOf(newSeq));
            ++realignedCount;
        }
        return realignedCount;
    }

    static boolean alignAsSameSequences(AlignmentI unaligned, AlignmentI aligned) {
        if (aligned.getDataset() == null || unaligned.getDataset() == null) {
            return false;
        }
        HashMap alignedDatasets = new HashMap();
        for (SequenceI sequenceI : aligned.getSequences()) {
            SequenceI ds = sequenceI.getDatasetSequence();
            if (alignedDatasets.get(ds) == null) {
                alignedDatasets.put(ds, new ArrayList());
            }
            ((List)alignedDatasets.get(ds)).add(sequenceI);
        }
        int leftmost = Integer.MAX_VALUE;
        for (SequenceI seq : unaligned.getSequences()) {
            SequenceI ds = seq.getDatasetSequence();
            if (!alignedDatasets.containsKey(ds)) {
                return false;
            }
            SequenceI alignedSeq = (SequenceI)((List)alignedDatasets.get(ds)).get(0);
            int startCol = alignedSeq.findIndex(seq.getStart());
            leftmost = Math.min(leftmost, startCol);
        }
        char c = aligned.getGapCharacter();
        for (SequenceI seq : unaligned.getSequences()) {
            List alignedSequences = (List)alignedDatasets.get(seq.getDatasetSequence());
            if (alignedSequences.isEmpty()) continue;
            SequenceI alignedSeq = (SequenceI)alignedSequences.get(0);
            int startCol = alignedSeq.findIndex(seq.getStart());
            int endCol = alignedSeq.findIndex(seq.getEnd());
            char[] seqchars = new char[endCol - leftmost + 1];
            Arrays.fill(seqchars, c);
            char[] toCopy = alignedSeq.getSequence(startCol - 1, endCol);
            System.arraycopy(toCopy, 0, seqchars, startCol - leftmost, toCopy.length);
            seq.setSequence(String.valueOf(seqchars));
            if (alignedSequences.size() <= 0) continue;
            alignedSequences.remove(0);
        }
        new RemoveGapColCommand("", unaligned.getSequencesArray(), 0, unaligned.getWidth() - 1, unaligned);
        return true;
    }

    static SortedMap<Integer, Map<SequenceI, Character>> buildMappedColumnsMap(AlignmentI unaligned, AlignmentI aligned, List<SequenceI> unmapped) {
        TreeMap<Integer, Map<SequenceI, Character>> map = new TreeMap<Integer, Map<SequenceI, Character>>();
        unmapped.addAll(unaligned.getSequences());
        List<AlignedCodonFrame> mappings = aligned.getCodonFrames();
        for (SequenceI seq : unaligned.getSequences()) {
            for (AlignedCodonFrame mapping : mappings) {
                Mapping seqMap;
                SequenceI fromSeq = mapping.findAlignedSequence(seq, aligned);
                if (fromSeq == null || !AlignmentUtils.addMappedPositions(seq, fromSeq, seqMap = mapping.getMappingBetween(fromSeq, seq), map)) continue;
                unmapped.remove(seq);
            }
        }
        return map;
    }

    static boolean addMappedPositions(SequenceI seq, SequenceI fromSeq, Mapping seqMap, Map<Integer, Map<SequenceI, Character>> map) {
        if (seqMap == null) {
            return false;
        }
        if (seqMap.getTo() == fromSeq.getDatasetSequence()) {
            seqMap = new Mapping(seq.getDatasetSequence(), seqMap.getMap().getInverse());
        }
        int toStart = seq.getStart();
        for (int[] fromRange : seqMap.getMap().getFromRanges()) {
            for (int i = 0; i < fromRange.length - 1; i += 2) {
                boolean forward = fromRange[i + 1] >= fromRange[i];
                int[] range = seqMap.locateMappedRange(fromRange[i], fromRange[i + 1]);
                if (range == null) {
                    Console.errPrintln("Error in mapping " + seqMap + " from " + fromSeq.getName());
                    return false;
                }
                int mappedCharPos = range[0];
                for (int fromCol = fromSeq.findIndex(fromRange[i]); mappedCharPos <= range[1] && fromCol <= fromSeq.getLength() && fromCol >= 0; fromCol += forward ? 1 : -1) {
                    if (Comparison.isGap(fromSeq.getCharAt(fromCol - 1))) continue;
                    Map<SequenceI, Character> seqsMap = map.get(fromCol);
                    if (seqsMap == null) {
                        seqsMap = new HashMap<SequenceI, Character>();
                        map.put(fromCol, seqsMap);
                    }
                    seqsMap.put(seq, Character.valueOf(seq.getCharAt(mappedCharPos - toStart)));
                    ++mappedCharPos;
                }
            }
        }
        return true;
    }

    public static boolean looksLikeEnsembl(AlignmentI alignment) {
        for (SequenceI seq : alignment.getSequences()) {
            String name = seq.getName();
            if (name.startsWith("ENSG") || name.startsWith("ENST")) continue;
            return false;
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static List<AlignmentAnnotation> getSecondaryStructureAnnots(AlignmentAnnotation[] annotations) {
        ArrayList<AlignmentAnnotation> ssAnnotations = new ArrayList<AlignmentAnnotation>();
        if (annotations == null || annotations.length == 0) {
            return null;
        }
        AlignmentAnnotation[] alignmentAnnotationArray = annotations;
        synchronized (annotations) {
            for (AlignmentAnnotation aa : annotations) {
                if (aa == null || aa.label == null || !Constants.SECONDARY_STRUCTURE_LABELS.containsKey(aa.label)) continue;
                ssAnnotations.add(aa);
            }
            // ** MonitorExit[var2_2] (shouldn't be in output)
            return ssAnnotations;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean isSecondaryStructurePresent(AlignmentAnnotation[] annotations) {
        boolean ssPresent = false;
        if (annotations == null || annotations.length == 0) {
            return false;
        }
        AlignmentAnnotation[] alignmentAnnotationArray = annotations;
        synchronized (annotations) {
            for (AlignmentAnnotation aa : annotations) {
                if (aa == null || aa.label == null || !Constants.SECONDARY_STRUCTURE_LABELS.containsKey(aa.label)) continue;
                ssPresent = true;
                break;
            }
            // ** MonitorExit[var2_2] (shouldn't be in output)
            return ssPresent;
        }
    }

    public static Color getSecondaryStructureAnnotationColour(char symbol) {
        if (symbol == 'C') {
            return Color.gray;
        }
        if (symbol == 'E') {
            return Color.green;
        }
        if (symbol == 'H') {
            return Color.red;
        }
        return Color.white;
    }

    public static char findSSAnnotationForGivenSeqposition(AlignmentAnnotation aa, int seqPosition) {
        int ss = 42;
        if (aa != null) {
            if (aa.getAnnotationForPosition(seqPosition) != null) {
                Annotation a = aa.getAnnotationForPosition(seqPosition);
                ss = a.secondaryStructure;
                if (ss == 32 || ss == 45) {
                    ss = 67;
                }
            } else {
                ss = 67;
            }
        }
        return (char)ss;
    }

    public static List<String> extractSSSourceInAlignmentAnnotation(AlignmentAnnotation[] annotations) {
        ArrayList<String> ssSources = new ArrayList<String>();
        HashSet<String> addedSources = new HashSet<String>();
        if (annotations == null) {
            return ssSources;
        }
        for (AlignmentAnnotation aa : annotations) {
            String ssSource = AlignmentAnnotationUtils.extractSSSourceFromAnnotationDescription(aa);
            if (ssSource == null || addedSources.contains(ssSource)) continue;
            ssSources.add(ssSource);
            addedSources.add(ssSource);
        }
        Collections.sort(ssSources);
        return ssSources;
    }

    public static List<AlignmentAnnotation> getAlignmentAnnotationForSource(SequenceI seq, String ssSource) {
        ArrayList<AlignmentAnnotation> ssAnnots = new ArrayList<AlignmentAnnotation>();
        for (String ssLabel : Constants.SECONDARY_STRUCTURE_LABELS.keySet()) {
            AlignmentAnnotation[] aa = seq.getAnnotation(ssLabel);
            if (aa == null) continue;
            if ("All".equals(ssSource)) {
                ssAnnots.addAll(Arrays.asList(aa));
                continue;
            }
            for (AlignmentAnnotation annot : aa) {
                String ssSourceForAnnot = AlignmentAnnotationUtils.extractSSSourceFromAnnotationDescription(annot);
                if (ssSourceForAnnot == null || !ssSource.equals(ssSourceForAnnot) || !annot.isForDisplay()) continue;
                ssAnnots.add(annot);
            }
        }
        if (ssAnnots.size() > 0) {
            return ssAnnots;
        }
        return null;
    }

    public static Map<SequenceI, ArrayList<AlignmentAnnotation>> getSequenceAssociatedAlignmentAnnotations(AlignmentAnnotation[] alignAnnotList, String selectedSSSource) {
        HashMap<SequenceI, ArrayList<AlignmentAnnotation>> ssAlignmentAnnotationForSequences = new HashMap<SequenceI, ArrayList<AlignmentAnnotation>>();
        if (alignAnnotList == null || alignAnnotList.length == 0) {
            return ssAlignmentAnnotationForSequences;
        }
        for (AlignmentAnnotation aa : alignAnnotList) {
            if (aa.sequenceRef == null || !AlignmentUtils.isSecondaryStructureFrom(selectedSSSource, aa)) continue;
            ssAlignmentAnnotationForSequences.computeIfAbsent(aa.sequenceRef.getDatasetSequence(), k -> new ArrayList()).add(aa);
        }
        return ssAlignmentAnnotationForSequences;
    }

    public static List<AlignmentAnnotation> getSecondaryStructureAnnotionFor(List<AlignmentAnnotation> alignAnnotation, String selectedSSSource) {
        ArrayList<AlignmentAnnotation> annForSource = new ArrayList<AlignmentAnnotation>();
        for (AlignmentAnnotation alan : alignAnnotation) {
            if (!AlignmentUtils.isSecondaryStructureFrom(selectedSSSource, alan)) continue;
            annForSource.add(alan);
        }
        return annForSource;
    }

    public static boolean isSecondaryStructureFrom(String selectedSSSource, AlignmentAnnotation aa) {
        for (String label : Constants.SECONDARY_STRUCTURE_LABELS.keySet()) {
            if (!label.equals(aa.label)) continue;
            if (selectedSSSource.equals("All")) {
                return true;
            }
            String ssSource = AlignmentAnnotationUtils.extractSSSourceFromAnnotationDescription(aa);
            if (ssSource == null || !ssSource.equals(selectedSSSource)) continue;
            return true;
        }
        return false;
    }

    public static String getSecondaryStructureProviderKey(String providerValue) {
        for (Map.Entry<String, String> entry : Constants.STRUCTURE_PROVIDERS.entrySet()) {
            if (!entry.getValue().equals(providerValue)) continue;
            return entry.getKey();
        }
        return null;
    }

    public static String reduceLabelLength(String label) {
        String[] parts = label.split(" \\| ");
        String reducedLabel = Arrays.stream(parts).map(fullName -> Constants.STRUCTURE_PROVIDERS.entrySet().stream().filter(entry -> ((String)entry.getValue()).equals(fullName)).map(Map.Entry::getKey).findFirst().orElse((String)fullName)).collect(Collectors.joining(" | "));
        return reducedLabel;
    }

    public static HashMap<String, Color> assignColorsForSecondaryStructureProviders(List<String> labels) {
        HashMap<String, Color> secondaryStructureProviderColorMap = new HashMap<String, Color>();
        for (String label : labels) {
            String name = label.toUpperCase(Locale.ROOT).trim();
            secondaryStructureProviderColorMap.put(name, ColorUtils.getColourFromNameAndScheme(name, "NONE"));
        }
        return secondaryStructureProviderColorMap;
    }

    public static int computeMaxShifts(SequenceI[] seqs, ShiftList inserts) {
        int oldwidth = 0;
        int p = 0;
        int lastP = 0;
        int lastIns = 0;
        for (SequenceI seq : seqs) {
            char[] sqs = seq.getSequence();
            if (oldwidth < sqs.length) {
                oldwidth = sqs.length;
            }
            p = 0;
            lastP = 0;
            lastIns = 0;
            int modelOffset = 0;
            do {
                if (sqs[p] >= 'a' && sqs[p] <= 'z') {
                    if (lastIns == 0) {
                        lastP = (modelOffset + p) * 2;
                    }
                    ++lastIns;
                    continue;
                }
                if (lastIns > 0) {
                    inserts.extendShift(lastP, lastIns);
                    modelOffset -= lastIns;
                }
                lastIns = 0;
            } while (++p < sqs.length);
            if (lastIns <= 0) continue;
            inserts.extendShift(lastP, lastIns);
        }
        return oldwidth;
    }

    public static String a3mToMSA(SequenceI[] seqs) {
        int newwidth;
        Object response = "";
        ShiftList inserts = new ShiftList();
        int oldwidth = AlignmentUtils.computeMaxShifts(seqs, inserts);
        if (oldwidth != (newwidth = AlignmentUtils.insertShifts(seqs, inserts))) {
            response = "Added " + (newwidth - oldwidth) + " inserts.";
        }
        return response;
    }

    public static int insertShifts(SequenceI[] seqs, ShiftList inserts) {
        int newwidth = 0;
        for (SequenceI seq : seqs) {
            boolean changed = false;
            int posShift = 0;
            int totalShift = 0;
            for (int[] shift : inserts.getShifts()) {
                int p;
                posShift = shift[0] / 2;
                int len = shift[1];
                for (p = posShift + totalShift; len > 0 && p < seq.getLength() && seq.getCharAt(p) >= 'a' && seq.getCharAt(p) <= 'z'; ++p, --len) {
                }
                if (len > 0) {
                    changed = true;
                    seq.doInsert(p, len, '-');
                }
                totalShift += shift[1];
            }
            if (changed) {
                seq.sequenceChanged();
            }
            newwidth = seq.getLength() > newwidth ? seq.getLength() : newwidth;
        }
        return newwidth;
    }

    static final class DnaVariant {
        final String base;
        SequenceFeature variant;

        DnaVariant(String nuc) {
            this.base = nuc;
            this.variant = null;
        }

        DnaVariant(String nuc, SequenceFeature var) {
            this.base = nuc;
            this.variant = var;
        }

        public String getSource() {
            return this.variant == null ? null : this.variant.getFeatureGroup();
        }

        public String toString() {
            return this.base + ":" + (this.variant == null ? "" : this.variant.getDescription());
        }
    }
}

