/*
 * Decompiled with CFR 0.152.
 */
package jalview.io.vcf;

import htsjdk.samtools.SAMException;
import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.SAMSequenceRecord;
import htsjdk.samtools.util.CloseableIterator;
import htsjdk.tribble.TribbleException;
import htsjdk.variant.variantcontext.Allele;
import htsjdk.variant.variantcontext.VariantContext;
import htsjdk.variant.vcf.VCFHeader;
import htsjdk.variant.vcf.VCFHeaderLine;
import htsjdk.variant.vcf.VCFHeaderLineCount;
import htsjdk.variant.vcf.VCFHeaderLineType;
import htsjdk.variant.vcf.VCFInfoHeaderLine;
import jalview.analysis.Dna;
import jalview.api.AlignViewControllerGuiI;
import jalview.bin.Cache;
import jalview.bin.Console;
import jalview.datamodel.DBRefEntry;
import jalview.datamodel.GeneLociI;
import jalview.datamodel.Mapping;
import jalview.datamodel.Sequence;
import jalview.datamodel.SequenceFeature;
import jalview.datamodel.SequenceI;
import jalview.datamodel.features.FeatureAttributeType;
import jalview.datamodel.features.FeatureSource;
import jalview.datamodel.features.FeatureSources;
import jalview.ext.ensembl.EnsemblMap;
import jalview.ext.htsjdk.HtsContigDb;
import jalview.ext.htsjdk.VCFReader;
import jalview.util.MapList;
import jalview.util.MappingUtils;
import jalview.util.MessageManager;
import jalview.util.StringUtils;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

public class VCFLoader {
    private static final String VCF_ENCODABLE = ":;=%,";
    private static final String VCF_POS = "POS";
    private static final String VCF_ID = "ID";
    private static final String VCF_QUAL = "QUAL";
    private static final String VCF_FILTER = "FILTER";
    private static final String NO_VALUE = ".";
    private static final String DEFAULT_SPECIES = "homo_sapiens";
    private static final String VEP_FIELDS_PREF = "VEP_FIELDS";
    private static final String VCF_FIELDS_PREF = "VCF_FIELDS";
    private static final String DEFAULT_VCF_FIELDS = ".*";
    private static final String DEFAULT_VEP_FIELDS = ".*";
    private static final String VCF_ASSEMBLY = "VCF_ASSEMBLY";
    private static final String DEFAULT_VCF_ASSEMBLY = "assembly19=GRCh37,hs37=GRCh37,grch37=GRCh37,grch38=GRCh38";
    private static final String VCF_SPECIES = "VCF_SPECIES";
    private static final String DEFAULT_REFERENCE = "grch37";
    private static final String CSQ_CONSEQUENCE_KEY = "Consequence";
    private static final String CSQ_ALLELE_KEY = "Allele";
    private static final String CSQ_ALLELE_NUM_KEY = "ALLELE_NUM";
    private static final String CSQ_FEATURE_KEY = "Feature";
    private static final String CSQ_FIELD = "CSQ";
    private static final String PIPE_REGEX = "\\|";
    private static final String COMMA = ",";
    private static final String FEATURE_GROUP_VCF = "VCF";
    private static final String EXCL = "!";
    protected String vcfFilePath;
    private Map<String, Map<int[], int[]>> assemblyMappings;
    private VCFReader reader;
    private VCFHeader header;
    private String vcfSpecies;
    private String vcfAssembly;
    private SAMSequenceDictionary dictionary;
    private int csqConsequenceFieldIndex = -1;
    private int csqAlleleFieldIndex = -1;
    private int csqAlleleNumberFieldIndex = -1;
    private int csqFeatureFieldIndex = -1;
    private String sourceId;
    List<String> vcfFieldsOfInterest;
    Map<Integer, String> vepFieldsOfInterest;
    private Set<String> badData;

    public VCFLoader(String vcfFile) {
        try {
            this.initialise(vcfFile);
        }
        catch (IOException e) {
            Console.errPrintln("Error opening VCF file: " + e.getMessage());
        }
        this.assemblyMappings = new HashMap<String, Map<int[], int[]>>();
    }

    public void loadVCF(final SequenceI[] seqs, final AlignViewControllerGuiI gui) {
        if (gui != null) {
            gui.setStatus(MessageManager.getString("label.searching_vcf"));
        }
        new Thread(){

            @Override
            public void run() {
                VCFLoader.this.doLoad(seqs, gui);
            }
        }.start();
    }

    public SequenceI loadVCFContig(String contig) {
        VCFHeaderLine headerLine = this.header.getOtherHeaderLine("reference");
        if (headerLine == null) {
            Console.error("VCF reference header not found");
            return null;
        }
        String ref = headerLine.getValue();
        if (ref.startsWith("file://")) {
            ref = ref.substring(7);
        }
        this.setSpeciesAndAssembly(ref);
        SequenceI seq = null;
        File dbFile = new File(ref);
        if (dbFile.exists()) {
            HtsContigDb db = new HtsContigDb("", dbFile);
            seq = db.getSequenceProxy(contig);
            this.loadSequenceVCF(seq);
            db.close();
        } else {
            Console.error("VCF reference not found: " + ref);
        }
        return seq;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void doLoad(SequenceI[] seqs, AlignViewControllerGuiI gui) {
        try {
            VCFHeaderLine ref = this.header.getOtherHeaderLine("reference");
            String reference = ref == null ? null : ref.getValue();
            this.setSpeciesAndAssembly(reference);
            int varCount = 0;
            int seqCount = 0;
            for (SequenceI seq : seqs) {
                int added = this.loadSequenceVCF(seq);
                if (added <= 0) continue;
                ++seqCount;
                varCount += added;
                this.transferAddedFeatures(seq);
            }
            if (gui != null) {
                String msg = MessageManager.formatMessage("label.added_vcf", varCount, seqCount);
                gui.setStatus(msg);
                if (gui.getFeatureSettingsUI() != null) {
                    gui.getFeatureSettingsUI().discoverAllFeatureData();
                }
            }
        }
        catch (Throwable e) {
            Console.errPrintln("Error processing VCF: " + e.getMessage());
            e.printStackTrace();
            if (gui != null) {
                gui.setStatus("Error occurred - see console for details");
            }
        }
        finally {
            if (this.reader != null) {
                try {
                    this.reader.close();
                }
                catch (IOException ref) {}
            }
            this.header = null;
            this.dictionary = null;
        }
    }

    protected void setSpeciesAndAssembly(String reference) {
        String[] tokens;
        if (reference == null) {
            Console.error("No VCF ##reference found, defaulting to grch37:homo_sapiens");
            reference = DEFAULT_REFERENCE;
        }
        reference = reference.toLowerCase(Locale.ROOT);
        String prop = Cache.getDefault(VCF_ASSEMBLY, DEFAULT_VCF_ASSEMBLY);
        for (String token : prop.split(COMMA)) {
            tokens = token.split("=");
            if (tokens.length != 2 || !reference.contains(tokens[0].trim().toLowerCase(Locale.ROOT))) continue;
            this.vcfAssembly = tokens[1].trim();
            break;
        }
        this.vcfSpecies = DEFAULT_SPECIES;
        prop = Cache.getProperty(VCF_SPECIES);
        if (prop != null) {
            for (String token : prop.split(COMMA)) {
                tokens = token.split("=");
                if (tokens.length != 2 || !reference.contains(tokens[0].trim().toLowerCase(Locale.ROOT))) continue;
                this.vcfSpecies = tokens[1].trim();
                break;
            }
        }
    }

    private void initialise(String filePath) throws IOException {
        this.vcfFilePath = filePath;
        this.reader = new VCFReader(filePath);
        this.header = this.reader.getFileHeader();
        try {
            this.dictionary = this.header.getSequenceDictionary();
        }
        catch (SAMException sAMException) {
            // empty catch block
        }
        this.sourceId = filePath;
        this.saveMetadata(this.sourceId);
        this.parseCsqHeader();
    }

    void saveMetadata(String theSourceId) {
        List<Pattern> vcfFieldPatterns = this.getFieldMatchers(VCF_FIELDS_PREF, ".*");
        this.vcfFieldsOfInterest = new ArrayList<String>();
        FeatureSource metadata = new FeatureSource(theSourceId);
        for (VCFInfoHeaderLine info : this.header.getInfoHeaderLines()) {
            String attributeId = info.getID();
            String desc = info.getDescription();
            VCFHeaderLineType type = info.getType();
            FeatureAttributeType attType = null;
            switch (type) {
                case Character: {
                    attType = FeatureAttributeType.Character;
                    break;
                }
                case Flag: {
                    attType = FeatureAttributeType.Flag;
                    break;
                }
                case Float: {
                    attType = FeatureAttributeType.Float;
                    break;
                }
                case Integer: {
                    attType = FeatureAttributeType.Integer;
                    break;
                }
                case String: {
                    attType = FeatureAttributeType.String;
                }
            }
            metadata.setAttributeName(attributeId, desc);
            metadata.setAttributeType(attributeId, attType);
            if (!this.isFieldWanted(attributeId, vcfFieldPatterns)) continue;
            this.vcfFieldsOfInterest.add(attributeId);
        }
        FeatureSources.getInstance().addSource(theSourceId, metadata);
    }

    private boolean isFieldWanted(String id, List<Pattern> filters) {
        for (Pattern p : filters) {
            if (!p.matcher(id.toUpperCase(Locale.ROOT)).matches()) continue;
            return true;
        }
        return false;
    }

    protected void parseCsqHeader() {
        List<Pattern> vepFieldFilters = this.getFieldMatchers(VEP_FIELDS_PREF, ".*");
        this.vepFieldsOfInterest = new HashMap<Integer, String>();
        VCFInfoHeaderLine csqInfo = this.header.getInfoHeaderLine(CSQ_FIELD);
        if (csqInfo == null) {
            return;
        }
        String desc = csqInfo.getDescription();
        int spacePos = desc.lastIndexOf(" ");
        if ((desc = desc.substring(spacePos + 1)) != null) {
            String[] format = desc.split(PIPE_REGEX);
            int index = 0;
            for (String field : format) {
                if (CSQ_CONSEQUENCE_KEY.equals(field)) {
                    this.csqConsequenceFieldIndex = index;
                }
                if (CSQ_ALLELE_NUM_KEY.equals(field)) {
                    this.csqAlleleNumberFieldIndex = index;
                }
                if (CSQ_ALLELE_KEY.equals(field)) {
                    this.csqAlleleFieldIndex = index;
                }
                if (CSQ_FEATURE_KEY.equals(field)) {
                    this.csqFeatureFieldIndex = index;
                }
                if (this.isFieldWanted(field, vepFieldFilters)) {
                    this.vepFieldsOfInterest.put(index, field);
                }
                ++index;
            }
        }
    }

    private List<Pattern> getFieldMatchers(String key, String def) {
        String[] tokens;
        String pref = Cache.getDefault(key, def);
        ArrayList<Pattern> patterns = new ArrayList<Pattern>();
        for (String token : tokens = pref.split(COMMA)) {
            try {
                patterns.add(Pattern.compile(token.toUpperCase(Locale.ROOT)));
            }
            catch (PatternSyntaxException e) {
                Console.errPrintln("Invalid pattern ignored: " + token);
            }
        }
        return patterns;
    }

    protected void transferAddedFeatures(SequenceI seq) {
        Sequence.DBModList<DBRefEntry> dbrefs = seq.getDBRefs();
        if (dbrefs == null) {
            return;
        }
        for (DBRefEntry dbref : dbrefs) {
            Mapping mapping = dbref.getMap();
            if (mapping == null || mapping.getTo() == null) continue;
            SequenceI mapTo = mapping.getTo();
            MapList map = mapping.getMap();
            if (map.getFromRatio() == 3) continue;
            List<SequenceFeature> features = seq.getFeatures().getPositionalFeatures("sequence_variant");
            for (SequenceFeature sf : features) {
                if (!FEATURE_GROUP_VCF.equals(sf.getFeatureGroup())) continue;
                this.transferFeature(sf, mapTo, map);
            }
        }
    }

    protected int loadSequenceVCF(SequenceI seq) {
        VCFMap vcfMap = this.getVcfMap(seq);
        if (vcfMap == null) {
            return 0;
        }
        SequenceI dss = seq.getDatasetSequence();
        if (dss == null) {
            dss = seq;
        }
        return this.addVcfVariants(dss, vcfMap);
    }

    private VCFMap getVcfMap(SequenceI seq) {
        VCFMap vcfMap = null;
        if (this.dictionary != null) {
            vcfMap = this.getContigMap(seq);
        }
        if (vcfMap != null) {
            return vcfMap;
        }
        GeneLociI seqCoords = seq.getGeneLoci();
        if (seqCoords == null) {
            Console.warn(String.format("Can't query VCF for %s as chromosome coordinates not known", seq.getName()));
            return null;
        }
        String species = seqCoords.getSpeciesId();
        String chromosome = seqCoords.getChromosomeId();
        String seqRef = seqCoords.getAssemblyId();
        MapList map = seqCoords.getMapping();
        if (!this.vcfSpecies.equalsIgnoreCase(species)) {
            Console.warn("No VCF loaded to " + seq.getName() + " as species not matched");
            return null;
        }
        if (seqRef.equalsIgnoreCase(this.vcfAssembly)) {
            return new VCFMap(chromosome, map);
        }
        ArrayList<int[]> toVcfRanges = new ArrayList<int[]>();
        ArrayList<int[]> fromSequenceRanges = new ArrayList<int[]>();
        for (int[] range : map.getToRanges()) {
            int[] fromRange = map.locateInFrom(range[0], range[1]);
            if (fromRange == null) continue;
            int[] newRange = this.mapReferenceRange(range, chromosome, "human", seqRef, this.vcfAssembly);
            if (newRange == null) {
                Console.error(String.format("Failed to map %s:%s:%s:%d:%d to %s", species, chromosome, seqRef, range[0], range[1], this.vcfAssembly));
                continue;
            }
            toVcfRanges.add(newRange);
            fromSequenceRanges.add(fromRange);
        }
        return new VCFMap(chromosome, new MapList(fromSequenceRanges, toVcfRanges, 1, 1));
    }

    private VCFMap getContigMap(SequenceI seq) {
        int len;
        String id = seq.getName();
        SAMSequenceRecord contig = this.dictionary.getSequence(id);
        if (contig != null && (len = seq.getLength()) == contig.getSequenceLength()) {
            MapList map = new MapList(new int[]{1, len}, new int[]{1, len}, 1, 1);
            return new VCFMap(id, map);
        }
        return null;
    }

    protected int addVcfVariants(SequenceI seq, VCFMap map) {
        boolean forwardStrand = map.map.isToForwardStrand();
        int count = 0;
        for (int[] range : map.map.getToRanges()) {
            int vcfStart = Math.min(range[0], range[1]);
            int vcfEnd = Math.max(range[0], range[1]);
            try {
                CloseableIterator<VariantContext> variants = this.reader.query(map.chromosome, vcfStart, vcfEnd);
                while (variants.hasNext()) {
                    VariantContext variant = (VariantContext)variants.next();
                    int[] featureRange = map.map.locateInFrom(variant.getStart(), variant.getEnd());
                    if (featureRange == null) continue;
                    int featureStart = Math.min(featureRange[0], featureRange[1]);
                    int featureEnd = Math.max(featureRange[0], featureRange[1]);
                    if (featureEnd - featureStart != variant.getEnd() - variant.getStart()) continue;
                    count += this.addAlleleFeatures(seq, variant, featureStart, featureEnd, forwardStrand);
                }
                variants.close();
            }
            catch (TribbleException e) {
                String msg = String.format("Error reading VCF for %s:%d-%d: %s ", map.chromosome, vcfStart, vcfEnd, e.getLocalizedMessage());
                Console.error(msg);
            }
        }
        return count;
    }

    protected String getAttributeValue(VariantContext variant, String attributeName, int alleleIndex) {
        Object att = variant.getAttribute(attributeName);
        if (att instanceof String) {
            return (String)att;
        }
        if (att instanceof ArrayList) {
            return (String)((List)att).get(alleleIndex);
        }
        return null;
    }

    protected int addAlleleFeatures(SequenceI seq, VariantContext variant, int featureStart, int featureEnd, boolean forwardStrand) {
        int added = 0;
        int altAlleleCount = variant.getAlternateAlleles().size();
        for (int i = 0; i < altAlleleCount; ++i) {
            added += this.addAlleleFeature(seq, variant, i, featureStart, featureEnd, forwardStrand);
        }
        return added;
    }

    protected int addAlleleFeature(SequenceI seq, VariantContext variant, int altAlleleIndex, int featureStart, int featureEnd, boolean forwardStrand) {
        String reference = variant.getReference().getBaseString();
        Allele alt = variant.getAlternateAllele(altAlleleIndex);
        Object allele = alt.getBaseString();
        int referenceLength = reference.length();
        if (!forwardStrand && ((String)allele).length() > referenceLength && ((String)allele).startsWith(reference)) {
            featureEnd = featureStart -= referenceLength;
            char insertAfter = seq.getCharAt(featureStart - seq.getStart());
            reference = Dna.reverseComplement(String.valueOf(insertAfter));
            allele = ((String)allele).substring(referenceLength) + reference;
        }
        StringBuilder sb = new StringBuilder();
        sb.append(forwardStrand ? reference : Dna.reverseComplement(reference));
        sb.append(COMMA);
        sb.append((String)(forwardStrand ? allele : Dna.reverseComplement((String)allele)));
        String alleles = sb.toString();
        String consequence = this.getConsequenceForAlleleAndFeature(variant, CSQ_FIELD, altAlleleIndex, this.csqAlleleFieldIndex, this.csqAlleleNumberFieldIndex, seq.getName().toLowerCase(Locale.ROOT), this.csqFeatureFieldIndex);
        String type = "sequence_variant";
        if (consequence != null) {
            type = this.getOntologyTerm(consequence);
        }
        SequenceFeature sf = new SequenceFeature(type, alleles, featureStart, featureEnd, FEATURE_GROUP_VCF);
        sf.setSource(this.sourceId);
        this.addFeatureAttribute(sf, "alleles", alleles);
        this.addFeatureAttribute(sf, VCF_POS, String.valueOf(variant.getStart()));
        this.addFeatureAttribute(sf, VCF_ID, variant.getID());
        this.addFeatureAttribute(sf, VCF_QUAL, String.valueOf(variant.getPhredScaledQual()));
        this.addFeatureAttribute(sf, VCF_FILTER, this.getFilter(variant));
        this.addAlleleProperties(variant, sf, altAlleleIndex, consequence);
        seq.addSequenceFeature(sf);
        return 1;
    }

    String getFilter(VariantContext variant) {
        Set filters = variant.getFilters();
        if (filters.isEmpty()) {
            return NO_VALUE;
        }
        Iterator iterator = filters.iterator();
        String first = (String)iterator.next();
        if (filters.size() == 1) {
            return first;
        }
        StringBuilder sb = new StringBuilder(first);
        while (iterator.hasNext()) {
            sb.append(";").append((String)iterator.next());
        }
        return sb.toString();
    }

    void addFeatureAttribute(SequenceFeature sf, String key, String value) {
        if (value != null && !value.isEmpty() && !NO_VALUE.equals(value)) {
            sf.setValue(key, value);
        }
    }

    String getOntologyTerm(String consequence) {
        int pos;
        String[] csqFields;
        String type = "sequence_variant";
        if (this.csqAlleleFieldIndex == -1) {
            return type;
        }
        if (consequence != null && (csqFields = consequence.split(PIPE_REGEX)).length > this.csqConsequenceFieldIndex) {
            type = csqFields[this.csqConsequenceFieldIndex];
        }
        if (type != null && (pos = type.indexOf(38)) > 0) {
            type = type.substring(0, pos);
        }
        return type;
    }

    private String getConsequenceForAlleleAndFeature(VariantContext variant, String vcfInfoId, int altAlleleIndex, int alleleFieldIndex, int alleleNumberFieldIndex, String seqName, int featureFieldIndex) {
        if (alleleFieldIndex == -1 || featureFieldIndex == -1) {
            return null;
        }
        Object value = variant.getAttribute(vcfInfoId);
        if (value == null || !(value instanceof List)) {
            return null;
        }
        List consequences = (List)value;
        for (String consequence : consequences) {
            String featureIdentifier;
            String[] csqFields = consequence.split(PIPE_REGEX);
            if (csqFields.length <= featureFieldIndex || (featureIdentifier = csqFields[featureFieldIndex]).length() <= 4 || seqName.indexOf(featureIdentifier.toLowerCase(Locale.ROOT)) <= -1 || !this.matchAllele(variant, altAlleleIndex, csqFields, alleleFieldIndex, alleleNumberFieldIndex)) continue;
            return consequence;
        }
        return null;
    }

    private boolean matchAllele(VariantContext variant, int altAlleleIndex, String[] csqFields, int alleleFieldIndex, int alleleNumberFieldIndex) {
        if (alleleNumberFieldIndex > -1) {
            if (csqFields.length <= alleleNumberFieldIndex) {
                return false;
            }
            String alleleNum = csqFields[alleleNumberFieldIndex];
            return String.valueOf(altAlleleIndex + 1).equals(alleleNum);
        }
        if (alleleFieldIndex > -1 && csqFields.length > alleleFieldIndex) {
            String csqAllele = csqFields[alleleFieldIndex];
            String vcfAllele = variant.getAlternateAllele(altAlleleIndex).getBaseString();
            return csqAllele.equals(vcfAllele);
        }
        return false;
    }

    protected void addAlleleProperties(VariantContext variant, SequenceFeature sf, int altAlelleIndex, String consequence) {
        Map atts = variant.getAttributes();
        for (Map.Entry att : atts.entrySet()) {
            String value;
            VCFInfoHeaderLine infoHeader;
            String key = (String)att.getKey();
            if (CSQ_FIELD.equals(key)) {
                this.addConsequences(variant, sf, consequence);
                continue;
            }
            if (!this.vcfFieldsOfInterest.contains(key) || (infoHeader = this.header.getInfoHeaderLine(key)) == null) continue;
            VCFHeaderLineCount number = infoHeader.getCountType();
            int index = altAlelleIndex;
            if (number == VCFHeaderLineCount.R) {
                ++index;
            } else if (number != VCFHeaderLineCount.A) continue;
            if ((value = this.getAttributeValue(variant, key, index)) == null || !this.isValid(variant, key, value)) continue;
            value = StringUtils.urlDecode(value, VCF_ENCODABLE);
            this.addFeatureAttribute(sf, key, value);
        }
    }

    protected boolean isValid(VariantContext variant, String infoId, String value) {
        if (value == null || value.isEmpty() || NO_VALUE.equals(value)) {
            return true;
        }
        VCFInfoHeaderLine infoHeader = this.header.getInfoHeaderLine(infoId);
        if (infoHeader == null) {
            Console.error("Field " + infoId + " has no INFO header");
            return false;
        }
        VCFHeaderLineType infoType = infoHeader.getType();
        try {
            if (infoType == VCFHeaderLineType.Integer) {
                Integer.parseInt(value);
            } else if (infoType == VCFHeaderLineType.Float) {
                Float.parseFloat(value);
            }
        }
        catch (NumberFormatException e) {
            this.logInvalidValue(variant, infoId, value);
            return false;
        }
        return true;
    }

    private void logInvalidValue(VariantContext variant, String infoId, String value) {
        String token;
        if (this.badData == null) {
            this.badData = new HashSet<String>();
        }
        if (!this.badData.contains(token = infoId + ":" + value)) {
            this.badData.add(token);
            Console.error(String.format("Invalid VCF data at %s:%d %s=%s", variant.getContig(), variant.getStart(), infoId, value));
        }
    }

    protected void addConsequences(VariantContext variant, SequenceFeature sf, String myConsequence) {
        Object value = variant.getAttribute(CSQ_FIELD);
        if (value == null || !(value instanceof List)) {
            return;
        }
        List consequences = (List)value;
        HashMap<String, String> csqValues = new HashMap<String, String>();
        for (String consequence : consequences) {
            if (myConsequence != null && !myConsequence.equals(consequence)) continue;
            String[] csqFields = consequence.split(PIPE_REGEX);
            int i = 0;
            for (String field : csqFields) {
                String id;
                if (field != null && field.length() > 0 && (id = this.vepFieldsOfInterest.get(i)) != null) {
                    field = StringUtils.urlDecode(field, VCF_ENCODABLE);
                    csqValues.put(id, field);
                }
                ++i;
            }
        }
        if (!csqValues.isEmpty()) {
            sf.setValue(CSQ_FIELD, csqValues);
        }
    }

    protected String complement(byte[] reference) {
        return String.valueOf(Dna.getComplement((char)reference[0]));
    }

    protected int[] mapReferenceRange(int[] queryRange, String chromosome, String species, String fromRef, String toRef) {
        int[] mappedRange = this.findSubsumedRangeMapping(queryRange, chromosome, species, fromRef, toRef);
        if (mappedRange != null) {
            return mappedRange;
        }
        EnsemblMap mapper = new EnsemblMap();
        int[] mapping = mapper.getAssemblyMapping(species, chromosome, fromRef, toRef, queryRange);
        if (mapping == null) {
            return null;
        }
        String key = VCFLoader.makeRangesKey(chromosome, species, fromRef, toRef);
        if (!this.assemblyMappings.containsKey(key)) {
            this.assemblyMappings.put(key, new HashMap());
        }
        this.assemblyMappings.get(key).put(queryRange, mapping);
        return mapping;
    }

    protected int[] findSubsumedRangeMapping(int[] queryRange, String chromosome, String species, String fromRef, String toRef) {
        String key = VCFLoader.makeRangesKey(chromosome, species, fromRef, toRef);
        if (this.assemblyMappings.containsKey(key)) {
            Map<int[], int[]> mappedRanges = this.assemblyMappings.get(key);
            for (Map.Entry<int[], int[]> mappedRange : mappedRanges.entrySet()) {
                int[] toRange;
                int[] fromRange = mappedRange.getKey();
                if (fromRange[1] - fromRange[0] != (toRange = mappedRange.getValue())[1] - toRange[0] || !MappingUtils.rangeContains(fromRange, queryRange)) continue;
                int offset = queryRange[0] - fromRange[0];
                int mappedRangeFrom = toRange[0] + offset;
                int mappedRangeTo = mappedRangeFrom + (queryRange[1] - queryRange[0]);
                return new int[]{mappedRangeFrom, mappedRangeTo};
            }
        }
        return null;
    }

    protected void transferFeature(SequenceFeature sf, SequenceI targetSequence, MapList mapping) {
        int[] mappedRange = mapping.locateInTo(sf.getBegin(), sf.getEnd());
        if (mappedRange != null) {
            String group = sf.getFeatureGroup();
            int newBegin = Math.min(mappedRange[0], mappedRange[1]);
            int newEnd = Math.max(mappedRange[0], mappedRange[1]);
            SequenceFeature copy = new SequenceFeature(sf, newBegin, newEnd, group, sf.getScore());
            targetSequence.addSequenceFeature(copy);
        }
    }

    protected static String makeRangesKey(String chromosome, String species, String fromRef, String toRef) {
        return species + EXCL + chromosome + EXCL + fromRef + EXCL + toRef;
    }

    class VCFMap {
        final String chromosome;
        final MapList map;

        VCFMap(String chr, MapList m) {
            this.chromosome = chr;
            this.map = m;
        }

        public String toString() {
            return this.chromosome + ":" + this.map.toString();
        }
    }
}

